/**
 *
 */
package de.upb.pga3.panda2.utilities;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.TreeSet;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.AnalysisGraph;
import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Transition;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisGraphLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.ParameterNode;
import de.upb.pga3.panda2.extension.lvl2a.TransitionLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.TransitionType;
import gnu.trove.map.TIntObjectMap;
import gnu.trove.map.TObjectIntMap;
import gnu.trove.map.hash.TIntObjectHashMap;
import gnu.trove.map.hash.TObjectIntHashMap;
import soot.Body;
import soot.SootClass;
import soot.SootMethod;
import soot.Unit;
import soot.toolkits.scalar.Pair;
import soot.util.HashMultiMap;
import soot.util.MultiMap;

/**
 * GraphGenerator class
 *
 * @author Fabian
 */
public class GraphPlotter {

	private static final Logger logger = LogManager.getLogger(GraphPlotter.class);

	private static GraphPlotter instance;

	public static GraphPlotter getInstance() {

		if (instance == null) {
			instance = new GraphPlotter();
		}
		return instance;
	}

	private final TObjectIntMap<Object> indexes = new TObjectIntHashMap<>();
	private final TIntObjectMap<String> labels = new TIntObjectHashMap<>();

	private static final String rSubName = "R_Class_Substitude";
	private static final int rSubIdx = 0;

	public void reset() {

		this.indexes.clear();
		this.labels.clear();
	}

	public void resetAndPlotGraph(final AnalysisGraphLvl2a g) {

		reset();
		plotGraph(g);
	}

	public void plotGraph(final AnalysisGraphLvl2a g) {

		final MultiMap<String, TransitionType> out = new HashMultiMap<>();
		out.putAll("graph_lvl2a_CD.gv", new HashSet<>(Arrays.asList(TransitionType.CONTROLDEPENDENCY)));
		out.putAll("graph_lvl2a_CF.gv", new HashSet<>(Arrays.asList(TransitionType.CONTROLFLOW)));
		out.putAll("graph_lvl2a_DF.gv", new HashSet<>(Arrays.asList(TransitionType.DATAFLOW)));
		out.putAll("graph_lvl2a_CALL.gv", new HashSet<>(Arrays.asList(TransitionType.CALL)));
		out.putAll("graph_lvl2a_PARAM.gv",
				new HashSet<>(Arrays.asList(TransitionType.PARAMIN, TransitionType.PARAMOUT)));
		out.putAll("graph_lvl2a_SUMMARY.gv", new HashSet<>(Arrays.asList(TransitionType.SUMMARY)));
		out.putAll("graph_lvl2a_CD_CF.gv",
				new HashSet<>(Arrays.asList(TransitionType.CONTROLDEPENDENCY, TransitionType.CONTROLFLOW)));
		out.putAll("graph_lvl2a_CALL_PARAM.gv",
				new HashSet<>(Arrays.asList(TransitionType.CALL, TransitionType.PARAMIN, TransitionType.PARAMOUT)));
		out.putAll("graph_lvl2a_CF_CALL_PARAM.gv", new HashSet<>(Arrays.asList(TransitionType.CONTROLFLOW,
				TransitionType.CALL, TransitionType.PARAMIN, TransitionType.PARAMOUT)));
		out.putAll("graph_lvl2a_ALL_NO_CF.gv",
				new HashSet<>(Arrays.asList(TransitionType.CONTROLDEPENDENCY, TransitionType.DATAFLOW,
						TransitionType.SUMMARY, TransitionType.CALL, TransitionType.PARAMIN, TransitionType.PARAMOUT)));
		out.putAll("graph_lvl2a_ALL.gv", new HashSet<>(Arrays.asList(TransitionType.values())));

		this.labels.put(GraphPlotter.rSubIdx, GraphPlotter.rSubName);

		for (final String file : out.keySet()) {
			final Path path = Paths.get(file).toAbsolutePath();
			try (BufferedWriter bw = Files.newBufferedWriter(path, StandardCharsets.UTF_8);) {
				plotGraph(bw, g, out.get(file));
			} catch (final IOException e) {
				e.printStackTrace();
				return;
			}
			try {
				final String[] dotCmd = { "dot", "-Tpdf", path.toString(), "-o", path.toString() + ".dot.pdf" };
				final Process p = Runtime.getRuntime().exec(dotCmd);
				p.waitFor();
			} catch (IOException | InterruptedException e) {
				logger.warn("Unable to plot graph with graphviz: {}", e.getMessage());
			}
			try {
				final String[] neatoCmd = { "neato", "-Goverlap=scale", "-Tpdf", path.toString(), "-o",
						path.toString() + ".neato.pdf" };
				final Process p = Runtime.getRuntime().exec(neatoCmd);
				p.waitFor();
			} catch (IOException | InterruptedException e) {
				logger.warn("Unable to plot graph with graphviz: {}", e.getMessage());
			}
		}
	}

	private void plotGraph(final Appendable app, final AnalysisGraph g, final Set<TransitionType> filter)
			throws IOException {

		app.append("digraph g {\n");
		app.append("edge [penwidth=2]\n");

		int numTrans = 0;
		final Set<Integer> drawnNodes = new TreeSet<>();
		Pair<Integer, Integer> sourceTargetIdx;
		final Set<Object> nodes = g.getNodes();
		logger.debug("Drawing outgoing transitions for {} nodes", nodes.size());
		final StringBuilder sb = new StringBuilder();
		for (final Object node : nodes) {
			for (final Transition t : g.getOutgoingTransitions(node)) {
				if (t instanceof TransitionLvl2a) {
					sourceTargetIdx = plotTransition(sb, g, (TransitionLvl2a) t, filter);
					if (sourceTargetIdx != null) {
						drawnNodes.add(sourceTargetIdx.getO1());
						drawnNodes.add(sourceTargetIdx.getO2());
						numTrans++;
					}
				}
			}
		}

		logger.debug("Drawing {} transitions for filters {}", numTrans, filter.toString());
		logger.debug("Drawing {} nodes for filters {}", drawnNodes.size(), filter.toString());
		for (final int idx : drawnNodes) {
			app.append(Integer.toString(idx));
			app.append(" [label=<");
			app.append(this.labels.get(idx));
			app.append(">];\n");
		}

		app.append(sb.toString());
		app.append("}");
	}

	private Pair<Integer, Integer> plotTransition(final Appendable app, final AnalysisGraph g,
			final TransitionLvl2a trans, final Set<TransitionType> filter) throws IOException {

		if (filter.contains(trans.getTransitionType())) {
			final int sourceIdx = getNodeIndex(trans.getSource(), g);
			final int targetIdx = getNodeIndex(trans.getTarget(), g);
			if (sourceIdx == GraphPlotter.rSubIdx && targetIdx == GraphPlotter.rSubIdx) {
				return null;
			}
			app.append(Integer.toString(sourceIdx));
			app.append(" -> ");
			app.append(Integer.toString(targetIdx));
			app.append(" [label=");
			app.append(trans.getTransitionType().name());
			String color = "black";
			switch (trans.getTransitionType()) {
			case CALL:
				color = "firebrick1";
				break;
			case DATAFLOW:
				color = "dodgerblue3";
				break;
			case SUMMARY:
				color = "gold3";
				break;
			case PARAMIN:
				color = "green3";
				break;
			case PARAMOUT:
				color = "green3";
				break;
			case CONTROLDEPENDENCY:
				color = "darkorange";
				break;
			case CONTROLFLOW:
				color = "darkorchid";
				break;
			default:
				color = "black";
				break;
			}
			app.append(",color=");
			app.append(color);
			app.append(",fontcolor=");
			app.append(color);
			app.append("];\n");
			return new Pair<>(sourceIdx, targetIdx);
		} else {
			return null;
		}

	}

	private int getNodeIndex(final Object node, final AnalysisGraph g) {

		if (this.indexes.containsKey(node)) {
			return this.indexes.get(node);
		}

		String unitStr = "-";
		String classStr = "-";
		String methodStr = "-";

		if (node instanceof Unit) {
			final EnhancedInput ei = (EnhancedInput) g.getInput();
			final Body body = ei.getBodyForUnit((Unit) node);
			classStr = body.getMethod().getDeclaringClass().toString();
			methodStr = body.getMethod().toString();
			unitStr = node.toString();

		} else if (node instanceof SootMethod) {
			classStr = ((SootMethod) node).getDeclaringClass().toString();
			methodStr = node.toString();
		} else if (node instanceof SootClass) {
			classStr = node.toString();
		} else if (node instanceof ParameterNode) {
			unitStr = "parameterNode" + node.hashCode();
		} else {
			throw new IllegalArgumentException("Unknown type: " + node.getClass());
		}

		classStr = classStr.replace("&", "&amp;").replace("\"", "&quot;").replace("<", "&lt;").replace(">", "&gt;");
		methodStr = methodStr.replace("&", "&amp;").replace("\"", "&quot;").replace("<", "&lt;").replace(">", "&gt;");
		unitStr = unitStr.replace("&", "&amp;").replace("\"", "&quot;").replace("<", "&lt;").replace(">", "&gt;");

		final String uName = classStr + "<BR/>" + methodStr + "<BR/>" + unitStr + "<BR/>" + node.hashCode();
		if (uName.contains(".R$")) {
			this.indexes.put(node, GraphPlotter.rSubIdx);
			return GraphPlotter.rSubIdx;
		}

		final int idx = this.labels.size();
		this.indexes.put(node, idx);
		this.labels.put(idx, uName);
		return idx;
	}

}
