package de.upb.pga3.panda2.utilities;

import gnu.trove.map.hash.TIntObjectHashMap;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.objenesis.strategy.StdInstantiatorStrategy;

import soot.PatchingChain;
import soot.SootClass;
import soot.Unit;
import soot.jimple.internal.JNopStmt;
import soot.jimple.internal.JTrap;
import soot.util.HashChain;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.ExternalizableSerializer;
import com.esotericsoftware.kryo.serializers.FieldSerializer;

import de.javakaffee.kryoserializers.UnmodifiableCollectionsSerializer;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;

/**
 * This class is used for serializing and de-serializing the instance of the
 * AnalysisResult class using the Kryo and its dependent libraries.
 *
 * @author Sriram
 *
 */
public class CustomSerializer {

	private static final Logger logger = LogManager.getLogger(CustomSerializer.class);

	/**
	 * Default Constructor
	 */
	public CustomSerializer() {
	}

	private static final Kryo kryo = new Kryo();

	static {
		kryo.setRegistrationRequired(false);

		// Registering the following class specific for level 1 and level 2b.
		kryo.register(TIntObjectHashMap.class, new ExternalizableSerializer());

		// Registering classes specific for level 2a.
		kryo.register(PatchingChain.class, new FieldSerializer(kryo, PatchingChain.class) {
			@Override
			public Object create(final Kryo kryo, final Input input, final Class type) {
				return new PatchingChain<Unit>(new HashChain<Unit>());
			}
		});
		final FieldSerializer test = new FieldSerializer(kryo, JTrap.class) {
			@Override
			public Object create(final Kryo kryo, final Input input, final Class type) {
				return new JTrap(new SootClass("test"), new JNopStmt(), new JNopStmt(), new JNopStmt());
			}
		};
		test.setCopyTransient(true);
		kryo.register(JTrap.class, test);
		UnmodifiableCollectionsSerializer.registerSerializers(kryo);

		((Kryo.DefaultInstantiatorStrategy) kryo.getInstantiatorStrategy())
				.setFallbackInstantiatorStrategy(new StdInstantiatorStrategy());
	}

	/**
	 * Method used for registering the known classes in Kryo. Used for improving
	 * the performance.
	 *
	 * @param classes
	 */
	public static void register(final Class... classes) {
		for (final Class clazz : classes) {
			kryo.register(clazz);
		}
	}

	/**
	 * Method used for saving the object in the file
	 *
	 * @param path
	 * @param anaResult
	 * @throws IOException
	 */
	public static void writeObject(final String path, final AnalysisResult anaResult) throws IOException {
		final FileOutputStream fos = new FileOutputStream(path);
		final Output output = new Output(fos);
		kryo.writeClassAndObject(output, anaResult);
		output.flush();
		output.close();
		fos.close();
	}

	/**
	 * Method used for reading the object from the file.
	 * 
	 * @param resultPath
	 * @return
	 * @throws IOException
	 */
	public static AnalysisResult readObject(final String resultPath) throws IOException {
		final FileInputStream fileIn = new FileInputStream(resultPath);
		final Input input = new Input(fileIn);
		final AnalysisResult analysisResult = (AnalysisResult) kryo.readClassAndObject(input);
		input.close();
		fileIn.close();
		return analysisResult;
	}
}
