package de.upb.pga3.panda2.extension.lvl2b.analyzer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.upb.pga3.panda2.core.datastructures.AnalysisGraph;
import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.Message;
import de.upb.pga3.panda2.core.datastructures.MessageType;
import de.upb.pga3.panda2.core.datastructures.Permission;
import de.upb.pga3.panda2.core.datastructures.ResultInput;
import de.upb.pga3.panda2.core.services.CoreServices;
import de.upb.pga3.panda2.extension.lvl1.AnalysisResultLvl1;
import de.upb.pga3.panda2.extension.lvl2b.AnalysisResultLvl2b;
import de.upb.pga3.panda2.extension.lvl2b.ResultItemLvl2b;
import de.upb.pga3.panda2.extension.lvl2b.ResultLeafLvl2b;
import de.upb.pga3.panda2.extension.lvl2b.ResultTreeLvl2b;
import de.upb.pga3.panda2.extension.lvl2b.ResultTypeLvl2b;
import soot.SootClass;

/**
 * This class computes the main part of any Level 2b analysis. It will primarily
 * be creating an {@link AnalysisResultLvl2b} object as analysis result.
 *
 * @author Felix
 *
 */
public class ManifestPermissionComparerLvl2b {
	private EnhancedInput currentEnhancedInput;

	private AnalysisGraph graph;
	private ResultInput ri;

	private Collection<Permission> allPermissions;

	private ResultLeafLvl2b resultApp;
	private ResultTreeLvl2b resultComponents;
	private AnalysisResultLvl2b anaResult;

	private boolean allmode;

	private Map<Object, List<Permission>> mapIndirectPermissions;
	private Map<Object, List<Permission>> mapBothPermissions;

	boolean flagMissingMessage = false;
	boolean flagMaybeMissingMessage = false;
	boolean flagUnusedMessage = false;
	boolean flagMaybeRequiredMessage = false;

	/**
	 * Constructor initializing the analysis result parts.
	 */
	public ManifestPermissionComparerLvl2b(final boolean allmode) {
		this.resultComponents = new ResultTreeLvl2b();
		this.allmode = allmode;
	}

	/**
	 * This method will be executed in order to determine the Permission-Groups
	 * of all elements of the analyzed App.
	 *
	 * @param inGraph
	 *            All information needed will be provided by this graph.
	 * @return The finished analysis result will be replied.
	 */
	public AnalysisResultLvl2b compare(final AnalysisGraph inGraph) {
		this.mapIndirectPermissions = new HashMap<>();
		this.mapBothPermissions = new HashMap<>();
		this.graph = inGraph;
		this.ri = (ResultInput) this.graph.getInput();
		this.currentEnhancedInput = (EnhancedInput) ((AnalysisResultLvl1) this.ri.getResults()
				.get(this.ri.getResults().size() - 1)).getAnalysisGraph().getInput();
		this.resultApp = new ResultLeafLvl2b(this.currentEnhancedInput.getAppName());
		this.allPermissions = CoreServices.getDataStorageInstance().getAllPermissions().values();
		final FixpointComputer fpc = new FixpointComputer(this.graph, this, this.allmode);
		fpc.computeFixpoint();
		this.anaResult = new AnalysisResultLvl2b();
		this.anaResult.initialize(CoreServices.getXMLParserInstance());
		generateAnalysisResult();
		for (int i = 0; i < this.ri.getResults().size(); i++) {
			final EnhancedInput ei = (EnhancedInput) ((AnalysisResultLvl1) this.ri.getResults().get(i))
					.getAnalysisGraph().getInput();
			if (!ei.getMaybeMoreList().isEmpty()) {
				this.anaResult.addMessage(new Message(MessageType.WARNING, "Target not found",
						"One or more targets of implicit intents could not be resolved."));
				break;
			}
		}
		return this.anaResult;
	}

	/**
	 * This method will add a {@link Permission} to an element. And mark it if
	 * it was indirectly accessed.
	 *
	 * @param object
	 *            The permission will be assigned to the element defined by this
	 *            parameter.
	 * @param permission
	 *            This {@link Permission} will be assigned to the element
	 *            specified by the first parameter.
	 * @param ei
	 *            The element belongs to the {@link EnhancedInput} specified in
	 *            this parameter.
	 */
	public void addIndirectPermission(final Object object, final Permission permission, final EnhancedInput ei) {
		if (!ei.getPermissions(object).contains(permission)) {
			ei.addPermissionTo(object, permission);

			// mark as indirect
			List<Permission> tempList = this.mapIndirectPermissions.get(object);
			if (tempList == null) {
				tempList = new ArrayList<>();
			}
			tempList.add(permission);
			this.mapIndirectPermissions.remove(object);
			this.mapIndirectPermissions.put(object, tempList);
		} else {
			// mark as indirect AND direct
			List<Permission> tempList = this.mapBothPermissions.get(object);
			if (tempList == null) {
				tempList = new ArrayList<>();
			}
			tempList.add(permission);
			this.mapBothPermissions.remove(object);
			this.mapBothPermissions.put(object, tempList);
		}
	}

	/**
	 * This method sorts the permissions into the 5 different groups
	 */
	private void generateAnalysisResult() {
		// Iterate over classes (components)
		for (final SootClass classOrComponentElement : this.currentEnhancedInput.getAppClasses()) {
			// Iterate over methods
			if (!classOrComponentElement.isInterface()) {
				checkElement(classOrComponentElement);
			}
		}
		checkElement(this.currentEnhancedInput);

		// Set partial results in result object
		this.anaResult.setApp(this.resultApp);
		this.anaResult.setComponents(this.resultComponents);
	}

	/**
	 * Assigns a Permission-Group to one specific element.
	 *
	 * @param element
	 *            The Permissin-Group will be assigned to this element.
	 */
	private void checkElement(final Object element) {
		// flags
		boolean hasPermission;
		boolean assignedToDescendantDirect;
		boolean assignedToDescendantIndirect;
		boolean inMaybeMore = false;

		for (final Permission permission : this.allPermissions) {
			// Check if app has current permission assigned
			if (this.currentEnhancedInput.getPermissions().contains(permission)) {
				hasPermission = true;
			} else {
				hasPermission = false;
			}

			// Check if any child has assigned that permission..
			assignedToDescendantDirect = false;
			assignedToDescendantIndirect = false;
			if (this.currentEnhancedInput.getChildren(element) != null) {
				for (final Object childElement : this.currentEnhancedInput.getChildren(element)) {
					if (assignedToChild(childElement, permission, false)) {
						assignedToDescendantDirect = true;
					}
					if (assignedToChild(childElement, permission, true)) {
						assignedToDescendantIndirect = true;
					}
					if (assignedToDescendantDirect && assignedToDescendantIndirect) {
						break;
					}
				}
			}

			// If needed, check if element is in maybeMore list
			if (!(hasPermission && (assignedToDescendantDirect || assignedToDescendantIndirect)
					|| !hasPermission && (assignedToDescendantDirect || assignedToDescendantIndirect))) {
				if (this.currentEnhancedInput.getMaybeMoreList().contains(element)) {
					inMaybeMore = true;
				}
			}

			// Set result accordingly to the set of flags
			setInResult(element, permission, hasPermission, assignedToDescendantDirect, assignedToDescendantIndirect,
					inMaybeMore);
		}
	}

	/**
	 * Checks whether the permission is assigned to any child of this child
	 * element.
	 *
	 * @param childElement
	 *            The element to investigate further.
	 * @param permission
	 *            The {@link Permission} to check
	 * @param checkForIndirect
	 *            Flag that determines if the child search should be accessed in
	 *            a direct or indirect way.
	 * @return If the {@link Permission} is assigned to any child true will be
	 *         returned. In any other case false.
	 */
	private boolean assignedToChild(final Object childElement, final Permission permission,
			final boolean checkForIndirect) {
		for (final Permission childPermission : this.currentEnhancedInput.getPermissions(childElement)) {
			if (childPermission == permission) {
				if (checkForIndirect) {
					if (this.mapIndirectPermissions.get(childElement) != null
							&& this.mapIndirectPermissions.get(childElement).contains(permission)) {
						return true;
					} else if (this.mapBothPermissions.get(childElement) != null
							&& this.mapBothPermissions.get(childElement).contains(permission)) {
						return true;
					}
				} else {
					if (this.mapIndirectPermissions.get(childElement) == null
							|| this.mapIndirectPermissions.get(childElement) != null
									&& !this.mapIndirectPermissions.get(childElement).contains(permission)) {
						return true;
					}
				}
			}
		}

		if (this.currentEnhancedInput.getChildren(childElement) != null) {
			for (final Object childsChildElement : this.currentEnhancedInput.getChildren(childElement)) {
				if (assignedToChild(childsChildElement, permission, checkForIndirect)) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Determines the Permission-Group.
	 *
	 * @param element
	 *            The Permission-Group will be determined for this element
	 * @param permission
	 *            The Permission-Group will be assigned to this
	 *            {@link Permission}.
	 * @param hasPermission
	 *            Flag representing the result of the check if the
	 *            {@link Permission} was assigned to this element.
	 * @param assignedToChildDirect
	 *            Flag representing the result of the check if the
	 *            {@link Permission} was assigned to any child of this element
	 *            (direct).
	 * @param assignedToChildIndirect
	 *            Flag representing the result of the check if the
	 *            {@link Permission} was assigned to any child of this element
	 *            (indirect).
	 * @param inMaybeMore
	 *            Flag representing the result of the check if this elements
	 *            belongs to the maybemore list or not.
	 */
	private void setInResult(final Object element, final Permission permission, final boolean hasPermission,
			final boolean assignedToChildDirect, final boolean assignedToChildIndirect, final boolean inMaybeMore) {
		int direct;
		if (assignedToChildDirect && assignedToChildIndirect) {
			direct = ResultItemLvl2b.BOTH;
		} else if (assignedToChildIndirect) {
			direct = ResultItemLvl2b.INDIRECT;
		} else {
			direct = ResultItemLvl2b.DIRECT;
		}

		if (hasPermission) {
			if (assignedToChildDirect || assignedToChildIndirect) {
				setInResult(element, permission, ResultTypeLvl2b.REQUIRED, direct);
				return;
			} else {
				if (inMaybeMore) {
					setInResult(element, permission, ResultTypeLvl2b.MAYBE_REQUIRED, direct);
					if (!this.flagMaybeRequiredMessage) {
						this.anaResult.addMessage(new Message(MessageType.WARNING, "MAYBE REQUIRED Permission",
								"The analyzed App contains one or more maybe required permissions. There might be a cooperation between this App and another that leads to a intended permission usage."));
						this.flagMaybeRequiredMessage = true;
					}
					return;
				} else {
					setInResult(element, permission, ResultTypeLvl2b.UNUSED, direct);
					if (!this.flagUnusedMessage) {
						this.anaResult.addMessage(new Message(MessageType.WARNING, "UNUSED Permission",
								"The analyzed App contains one or more unused permissions. Permissions are declared in the Android manifest but never used."));
						this.flagUnusedMessage = true;
					}
					return;
				}
			}
		} else {
			if (assignedToChildDirect || assignedToChildIndirect) {
				setInResult(element, permission, ResultTypeLvl2b.MISSING, direct);
				if (!this.flagMissingMessage) {
					this.anaResult.addMessage(new Message(MessageType.ERROR, "MISSING Permission",
							"The analyzed App contains one or more missing permissions. This should be considered as security critical and might lead to a data leak."));
					this.flagMissingMessage = true;
				}
				return;
			} else {
				if (inMaybeMore) {
					setInResult(element, permission, ResultTypeLvl2b.MAYBE_MISSING, direct);
					if (!this.flagMaybeMissingMessage) {
						this.anaResult.addMessage(new Message(MessageType.WARNING, "MAYBE MISSING Permission",
								"The analyzed App contains one or more maybe missing permissions. There might be a cooperation between this App and another that leads to a unintended permission usage."));
						this.flagMaybeMissingMessage = true;
					}
					return;
				}
			}
		}
	}

	/**
	 * Will set the final Permission-Group for this pair of element and
	 * {@link Permission}.
	 *
	 * @param element
	 *            The Permission-Group will be determined for this element
	 * @param permission
	 *            The Permission-Group will be assigned to this
	 *            {@link Permission}.
	 * @param group
	 *            The determined Permission-Group.
	 * @param direct
	 *            Type of {@link Permission} access: direct or indirect.
	 */
	private void setInResult(final Object element, final Permission permission, final int group, final int direct) {
		if (element instanceof EnhancedInput) {
			this.resultApp.addPermission(permission, group, direct);
		} else if (element instanceof SootClass) {
			if (this.currentEnhancedInput.isAndroidComponent((SootClass) element)) {
				ResultLeafLvl2b tempLeaf = this.resultComponents.getLeaf(element.toString());
				if (tempLeaf == null) {
					tempLeaf = new ResultLeafLvl2b(element.toString());
					this.resultComponents.addLeaf(tempLeaf);
				}
				tempLeaf.addPermission(permission, group, direct);
			}
		}
	}
}
