package de.upb.pga3.panda2.extension.lvl2a.graphgenerator;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.SootAdapter;
import de.upb.pga3.panda2.core.SootAdapter.SootPhase;
import de.upb.pga3.panda2.core.services.ARSCParser;
import de.upb.pga3.panda2.core.services.ARSCParser.AbstractResource;
import de.upb.pga3.panda2.core.services.ARSCParser.StringResource;
import de.upb.pga3.panda2.core.services.CoreServices;
import de.upb.pga3.panda2.core.services.XMLLayoutParser;
import de.upb.pga3.panda2.utilities.Constants;
import soot.ArrayType;
import soot.Body;
import soot.BooleanType;
import soot.ByteType;
import soot.CharType;
import soot.G;
import soot.Local;
import soot.Modifier;
import soot.RefType;
import soot.Scene;
import soot.ShortType;
import soot.SootClass;
import soot.SootMethod;
import soot.Transform;
import soot.Type;
import soot.Value;
import soot.VoidType;
import soot.dava.internal.javaRep.DIntConstant;
import soot.javaToJimple.LocalGenerator;
import soot.jimple.AssignStmt;
import soot.jimple.DoubleConstant;
import soot.jimple.FloatConstant;
import soot.jimple.IntConstant;
import soot.jimple.InvokeExpr;
import soot.jimple.Jimple;
import soot.jimple.JimpleBody;
import soot.jimple.LongConstant;
import soot.jimple.NewArrayExpr;
import soot.jimple.NewExpr;
import soot.jimple.NullConstant;
import soot.jimple.Stmt;
import soot.jimple.StringConstant;
import soot.jimple.toolkits.scalar.NopEliminator;
import soot.options.Options;

/**
 *
 * @author RamKumar
 *
 */

/**
 *
 * The Implementation is based in the implementation of Dummy main of FlowDroid
 *
 */

public class DummyMain {

	private static final Logger LOGGER = LogManager.getLogger(DummyMain.class);
	private final Set<String> entryPoints = new HashSet<>();
	private SootMethod mainMethod;
	private LocalGenerator generator;

	private JimpleBody body;
	private final String dummyClassName = "dummyMainClass";
	private final String dummyMethodName = "dummyMain";
	SootClass applicationClass = null;
	private final Set<SootClass> failedClasses = new HashSet<>();
	private final Map<String, Local> localVarClass = new HashMap<>();
	private final Set<SootMethod> failedMethods = new HashSet<>();

	private final Set<SootClass> applCallbkClasses = new HashSet<>();
	private Local applicationLocal = null;
	private Map<String, List<SootMethod>> callbackFunctions = new HashMap<>();
	private Map<String, List<Integer>> actvtyLyoutMap = new HashMap<>();
	private Map<String, List<String>> layoutCallbackMap = new HashMap<>();
	private static final boolean DEBUG = false;

	private enum componentType {
		Dummy, Application, Activity, ContentProvider, Service, BroadcastReceiver;
	}

	/**
	 * Creates empty Dummy Main method. Created dummy main is passed to
	 * createDummyMainBody method method
	 *
	 * @return Dummy Main Method to be used for Call graph generation.
	 */

	public SootMethod initialize() {
		getEntryPoints();
		/* Get android list of Android call back classes from A3DataStorage */
		final Set<String> androidCallBacks = CoreServices.getDataStorageInstance().getCallBackClasses();

		/*
		 * Initialise Call Back analyser and add it to soot transformation pack
		 */
		final CallBackAnalyser cbAny = CallBackAnalyser.getInstance();
		cbAny.setInputs(androidCallBacks, this.entryPoints);
		SootAdapter.getInstance().addTransformer(new Transform("wjtp.callbkanly", cbAny), SootPhase.WJTP);

		/* Initialise layout parser and add it to soot transformation pack */
		final XMLLayoutParser layoutParser = CoreServices.getXMLParserInstance().getLayoutParser();

		/* Get Arscparser instance from Core services */
		final ARSCParser arscParser = CoreServices.getXMLParserInstance().getArscParser();

		/* Run Soot */
		SootAdapter.getInstance().addTransformer(new Transform("wjtp.layoutpars", layoutParser), SootPhase.WJTP);
		SootAdapter.getInstance().run(SootPhase.WJTP);

		/* Get call back, layout ID mappings from soot */
		this.callbackFunctions = cbAny.getCallBackMethods();
		this.actvtyLyoutMap = cbAny.getActivityLayoutIDMap();
		this.layoutCallbackMap = layoutParser.getLayoutCallbackMethods();

		/*
		 * Identify layout call back methods and add to list of call back
		 * methods
		 */
		addlayoutCallBackMethods(arscParser);

		/* Create an empty dummy main */
		final SootMethod emptyMain = createDummyMain(Jimple.v().newBody());

		/* add statements to empty dummy main body */
		this.mainMethod = createMainBody(emptyMain);

		return this.mainMethod;

	}

	/**
	 * gets a list of Entry Points(Component classes) obtained from Manifest
	 * Parser.
	 *
	 */
	private void getEntryPoints() {

		/*
		 * ====================================================================
		 * cover cases for activities, services, receivers and provider
		 * ====================================================================
		 */
		final List<String> lstActivities = CoreServices.getXMLParserInstance().getLstActivities();

		final List<String> lstProviders = CoreServices.getXMLParserInstance().getLstProviders();

		final List<String> lstReceivers = CoreServices.getXMLParserInstance().getLstReceivers();

		final List<String> lstServices = CoreServices.getXMLParserInstance().getLstServices();

		// activities
		if (lstActivities != null) {
			for (final String actName : lstActivities) {
				this.entryPoints.add(actName);
			}
		}

		// content providers
		if (lstProviders != null) {
			for (final String proName : lstProviders) {
				this.entryPoints.add(proName);
			}
		}
		// broadcast receivers
		if (lstReceivers != null) {
			for (final String recName : lstReceivers) {
				this.entryPoints.add(recName);
			}
		}
		// services
		if (lstServices != null) {
			for (final String serName : lstServices) {
				this.entryPoints.add(serName);
			}
		}

	}

	/**
	 * To create a dummy Main class and dummy Main method, add dummy Main to the
	 * dummy Main class.
	 *
	 * @param b
	 *            - Body for the dummy Main method
	 * @return dummy Main class with dummy Main method.
	 */

	private SootMethod createDummyMain(final Body mainBody) {

		final SootClass dummyClass = new SootClass(this.dummyClassName);
		Scene.v().addClass(dummyClass);
		dummyClass.setApplicationClass();

		final Type stringArrayType = ArrayType.v(RefType.v("java.lang.String"), 1);
		final SootMethod dummyMain = new SootMethod(this.dummyMethodName, Collections.singletonList(stringArrayType),
				VoidType.v(), Modifier.PUBLIC | Modifier.STATIC);
		mainBody.setMethod(dummyMain);
		dummyMain.setActiveBody(mainBody);
		dummyClass.addMethod(dummyMain);
		final LocalGenerator locgen = new LocalGenerator(mainBody);
		final Local paramLocal = locgen.generateLocal(stringArrayType);
		mainBody.getUnits()
				.addFirst(Jimple.v().newIdentityStmt(paramLocal, Jimple.v().newParameterRef(stringArrayType, 0)));
		dummyClass.setApplicationClass();
		return dummyMain;
	}

	/**
	 * Add invoke statements to Android application classes and lifecycle,call
	 * back methods
	 *
	 * @param emptySootMethod
	 *            empty soot method body
	 * @return complete dummy main method body
	 */

	private SootMethod createMainBody(final SootMethod emptySootMethod) {

		/**
		 * Parsing the class names to Jimple format-
		 */
		this.mainMethod = emptySootMethod;
		this.body = (JimpleBody) this.mainMethod.getActiveBody();
		this.generator = new LocalGenerator(this.body);

		for (final String className : this.entryPoints) {
			Scene.v().forceResolve(className, SootClass.SIGNATURES);
		}

		/**
		 * In Android Lifecycle, OnCreate() of Content Providers are invoked
		 * first. Creating Jimple statements to invoke OnCreate() of Content
		 * Providers, if present
		 */

		{

			for (final String sootClassName : this.entryPoints) {
				final SootClass currentClass = Scene.v().getSootClass(sootClassName);

				if (getComponentType(currentClass) == componentType.ContentProvider) {

					// creating instance for the component class

					final Local contentLoc = generateClassConstructor(currentClass, this.body);

					if (contentLoc == null) {
						LOGGER.warn("Unable to create constructor for {}", currentClass.getName());
					} else {
						this.localVarClass.put(currentClass.getName(), contentLoc);
					}

					// creating invoke statement for onCreate of content
					// provider class

					final SootMethod currentMethod = findMethod(currentClass, Constants.CONTENTPROVIDER_ONCREATE);

					buildMethodCall(currentMethod, this.body, contentLoc, this.generator);

				}

			}
		}

		/**
		 * Now If there are any Android Application classes, they should be
		 * invoked
		 */

		{

			for (final String className : this.entryPoints) {

				final SootClass currentClass = Scene.v().getSootClass(className);
				final List<SootClass> parentClasses = Scene.v().getActiveHierarchy().getSuperclassesOf(currentClass);
				for (final SootClass sClass : parentClasses) {
					if (sClass.getName().contains("android.app.Application")) {
						if (this.applicationClass != null) {
							throw new RuntimeException("App has multiple application classes");
						}
						this.applicationClass = currentClass;
						this.applCallbkClasses.add(currentClass);

						// creating constructor for application class

						this.applicationLocal = generateClassConstructor(this.applicationClass, this.body);
						if (this.applicationLocal == null) {
							LOGGER.warn("Constructor cannot be created for {}", this.applicationClass);
						} else {
							this.localVarClass.put(this.applicationClass.getName(), this.applicationLocal);
						}

						// Adding invoke statements for callback methods in
						// application class

						if (this.callbackFunctions.containsKey(this.applicationClass.getName())) {

							for (final SootMethod callBackfn : this.callbackFunctions
									.get(this.applicationClass.getName())) {

								final String callBackClass = callBackfn.getDeclaringClass().getName();

								Local loc = this.localVarClass.get(callBackClass);
								if (loc == null) {
									final SootClass sClass1 = Scene.v().getSootClass(callBackClass);
									this.applCallbkClasses.add(sClass1);
									loc = generateClassConstructor(sClass1, this.body,
											Collections.singleton(this.applicationClass));
									if (loc != null) {
										this.localVarClass.put(callBackClass, loc);
									} else {
										LOGGER.warn("Unable to create constructor for {}", callBackClass);
									}
								}

							}

							// Calling OnCreate method of the Application class

							searchAndBuildMethod("void onCreate()", this.applicationClass, this.entryPoints,
									this.applicationLocal);
							break;

						}

					}

				}
			}
		}

		/*
		 * Check Android components and add invoke statements for constructors
		 * and life cycle, call back methods
		 */

		for (final String className : this.entryPoints) {

			final SootClass currentClass = Scene.v().getSootClass(className);
			currentClass.setApplicationClass();

			final componentType componentType = getComponentType(currentClass);

			if (!this.localVarClass.containsKey(currentClass.getName())) {
				final Local localVal = generateClassConstructor(currentClass, this.body);
				if (localVal == null) {
					LOGGER.warn("Constructor not created for {} ", currentClass.getName());
				} else {
					this.localVarClass.put(currentClass.getName(), localVal);
				}
			}

			final Local classLocal = this.localVarClass.get(currentClass.getName());

			switch (componentType) {
			case Activity:
				/*
				 * add invoke statements for Activity classes and their methods
				 */
				generateActivityLifecycle(this.entryPoints, currentClass, classLocal);
				break;
			case Service:
				/*
				 * add invoke statements for Service classes and their methods
				 */
				generateServiceLifecycle(this.entryPoints, currentClass, classLocal);
				break;
			case BroadcastReceiver:
				/*
				 * add invoke statements for Broadcast Receiver classes and
				 * their methods
				 */
				generateBroadcastReceiverLifecycle(this.entryPoints, currentClass, classLocal);
				break;
			case ContentProvider:
				/*
				 * add invoke statements for Content Provider classes and their
				 * methods
				 */

				generateContentProviderLifecycle(this.entryPoints, currentClass, classLocal);
				break;
			default:
				// Skip the entire class
				break;

			}

		}

		this.body.getUnits().add(Jimple.v().newReturnVoidStmt());

		NopEliminator.v().transform(this.body);

		if (DEBUG || Options.v().validate()) {
			this.mainMethod.getActiveBody().validate();
		}
		return this.mainMethod;
	}

	/**
	 * This method finds the Super class for a given Soot Class and determines
	 * the Android component type associated with that class
	 *
	 * @param sClass
	 *            - SootClass for which component type is to be determined.
	 * @return - the component type for the given class.
	 */

	private componentType getComponentType(final SootClass sClass) {

		componentType cType = componentType.Dummy;
		final List<SootClass> superClasses = Scene.v().getActiveHierarchy().getSuperclassesOf(sClass);
		for (final SootClass superClass : superClasses) {
			final String superClassName = superClass.getName();
			if (superClassName.contains("Activity")) {
				cType = componentType.Activity;
			} else if (superClassName.contains("Service")) {
				cType = componentType.Service;
			} else if (superClassName.contains("ContentProvider")) {
				cType = componentType.ContentProvider;
			} else if (superClassName.contains("BroadcastReceiver")) {
				cType = componentType.BroadcastReceiver;
			} else {
				cType = componentType.Dummy;
			}
			break;

		}
		return cType;
	}

	/**
	 * Generate invoke statement for class constructor
	 *
	 * @param createdClass
	 *            - Class to be invoked
	 * @param body
	 *            - Dummy main method body
	 * @return - Class local variable
	 */
	private Local generateClassConstructor(final SootClass createdClass, final Body body) {

		return this.generateClassConstructor(createdClass, body, new HashSet<SootClass>(),
				Collections.<SootClass> emptySet());
	}

	/**
	 * Generate invoke statement for class constructor
	 *
	 * @param createdClass
	 *            - Class to be invoked
	 * @param body
	 *            - Dummy main method body
	 * @param parentClasses
	 *            - Parent classes for the given class
	 * @return - Class local variable
	 */
	private Local generateClassConstructor(final SootClass createdClass, final Body body,
			final Set<SootClass> parentClasses) {

		return this.generateClassConstructor(createdClass, body, new HashSet<SootClass>(), parentClasses);
	}

	/**
	 * Generate invoke statement for class constructor
	 *
	 * @param createdClass
	 *            - Class to be invoked
	 * @param body
	 *            - Dummy main method body
	 * @param constructionStack
	 *            - Construction stack list of classes
	 * @param parentClasses
	 *            - Parent classes for the given class
	 * @return - Class local variable
	 */
	private Local generateClassConstructor(final SootClass createdClass, final Body body,
			final Set<SootClass> constructionStack, final Set<SootClass> parentClasses) {

		if (createdClass == null || this.failedClasses.contains(createdClass)) {
			return null;
		}

		/* For Phantom classes, constructors will not to be created. */

		if (createdClass.isPhantom() || createdClass.isPhantomClass()) {
			this.failedClasses.add(createdClass);
			return null;
		}

		/*
		 * Check if the class type is simple java type, create class variable
		 * and assignment statement
		 */
		if (isSimpleType(createdClass.toString())) {
			final Local varLocal = this.generator.generateLocal(getSimpleTypeFromType(createdClass.getType()));

			final AssignStmt aStmt = Jimple.v().newAssignStmt(varLocal, getSimpleDefaultValue(createdClass.toString()));
			body.getUnits().add(aStmt);
			return varLocal;
		}

		/*
		 * Check if the class is inner class of any other class, we create class
		 * variable and assignment statement
		 */
		final boolean isInnerClass = createdClass.getName().contains("$");
		final String outerClass = isInnerClass
				? createdClass.getName().substring(0, createdClass.getName().lastIndexOf("$")) : "";

		if (!constructionStack.add(createdClass)) {
			final Local tempLocal = this.generator.generateLocal(RefType.v(createdClass));
			final AssignStmt assignStmt = Jimple.v().newAssignStmt(tempLocal, NullConstant.v());
			body.getUnits().add(assignStmt);
			return tempLocal;
		}

		else {
			/* We check for explicit constructor methods */
			for (final SootMethod currentMethod : createdClass.getMethods()) {
				if (currentMethod.isPrivate() || !currentMethod.isConstructor()) {
					continue;
				}

				final List<Value> params = new LinkedList<>();
				for (final Type type : currentMethod.getParameterTypes()) {
					/*
					 * We check whether we have a reference to the outer class.
					 * In this case, we do not generate a new instance, but use
					 * the one we already have.
					 */
					final String typeName = type.toString().replaceAll("\\[\\]]", "");
					if (type instanceof RefType && isInnerClass && typeName.equals(outerClass)
							&& this.localVarClass.containsKey(typeName)) {
						params.add(this.localVarClass.get(typeName));
					} else {
						params.add(getValueForType(body, this.generator, type, constructionStack, parentClasses));
					}
				}

				/* Creating class variable and assignment statement */
				final NewExpr newExpr = Jimple.v().newNewExpr(RefType.v(createdClass));
				final Local tempLocal = this.generator.generateLocal(RefType.v(createdClass));
				final AssignStmt assignStmt = Jimple.v().newAssignStmt(tempLocal, newExpr);
				body.getUnits().add(assignStmt);

				/* create invoke statements for explicit constructors */
				InvokeExpr vInvokeExpr;
				if (params.isEmpty() || params.contains(null)) {
					vInvokeExpr = Jimple.v().newSpecialInvokeExpr(tempLocal, currentMethod.makeRef());
				} else {
					vInvokeExpr = Jimple.v().newSpecialInvokeExpr(tempLocal, currentMethod.makeRef(), params);
				}

				/*
				 * if the method is not void type, we create local variable for
				 * the return statement
				 */
				if (!(currentMethod.getReturnType() instanceof VoidType)) {
					final Local possibleReturn = this.generator.generateLocal(currentMethod.getReturnType());
					final AssignStmt assignStmt2 = Jimple.v().newAssignStmt(possibleReturn, vInvokeExpr);
					body.getUnits().add(assignStmt2);
				} else {
					body.getUnits().add(Jimple.v().newInvokeStmt(vInvokeExpr));
				}

				return tempLocal;
			}

			this.failedClasses.add(createdClass);
			return null;
		}
	}

	/**
	 * Method checks if the class type is a simple java type
	 *
	 * @param type
	 *            class type
	 * @return true if simple type, false otherwise
	 */

	private static boolean isSimpleType(final String type) {

		return Constants.METHOD_TYPES.contains(type);

	}

	/**
	 * Method returns soot equivalent of java simple types
	 *
	 * @param type
	 *            java class type
	 * @return soot type
	 */
	private Type getSimpleTypeFromType(final Type type) {
		if (type.toString().equals("java.lang.String")) {
			assert type instanceof RefType;
			return RefType.v(((RefType) type).getSootClass());
		}
		if (type.toString().equals("void")) {
			return VoidType.v();
		}
		if (type.toString().equals("char")) {
			return CharType.v();
		}
		if (type.toString().equals("byte")) {

			return ByteType.v();
		}
		if (type.toString().equals("short")) {
			return ShortType.v();
		}
		if (type.toString().equals("int")) {
			return soot.IntType.v();
		}
		if (type.toString().equals("float")) {
			return soot.FloatType.v();
		}
		if (type.toString().equals("long")) {
			return soot.LongType.v();
		}
		if (type.toString().equals("double")) {
			return soot.DoubleType.v();
		}
		if (type.toString().equals("boolean")) {
			return BooleanType.v();
		}
		throw new RuntimeException("Unknown simple type: " + type);
	}

	/**
	 *
	 * @param type
	 *            java class type
	 * @return Soot type value
	 */
	private Value getSimpleDefaultValue(final String type) {
		if ("java.lang.String".equals(type)) {
			return StringConstant.v("");
		}
		if ("char".equals(type)) {
			return DIntConstant.v(0, CharType.v());
		}
		if ("byte".equals(type)) {
			return DIntConstant.v(0, ByteType.v());
		}
		if ("short".equals(type)) {
			return DIntConstant.v(0, ShortType.v());
		}
		if ("int".equals(type)) {
			return IntConstant.v(0);
		}
		if ("float".equals(type)) {
			return FloatConstant.v(0);
		}
		if ("long".equals(type)) {
			return LongConstant.v(0);
		}
		if ("double".equals(type)) {
			return DoubleConstant.v(0);
		}
		if ("boolean".equals(type)) {
			return DIntConstant.v(0, BooleanType.v());
		}

		/* for arrays and other types, we return null */
		return G.v().soot_jimple_NullConstant();
	}

	/**
	 * Find the Soot method object in the given class using given method name
	 *
	 * @param sClass
	 *            class
	 * @param subsignature
	 *            name of method to be found in the class
	 * @return SootMethod corresponding to the method
	 */
	private SootMethod findMethod(final SootClass sClass, final String subsignature) {
		if (sClass.declaresMethod(subsignature)) {
			return sClass.getMethod(subsignature);
		}
		if (sClass.hasSuperclass()) {
			findMethod(sClass.getSuperclass(), subsignature);
		}
		return null;
	}

	/**
	 * Create invoke statement to call a method
	 *
	 * @param methodToCall
	 *            - method to be invoked
	 * @param body
	 *            - dummy main method body
	 * @param classLocal
	 *            - class variable
	 * @param gen
	 *            - soot local generator object
	 * @return - Soot Stmt
	 */
	private Stmt buildMethodCall(final SootMethod methodToCall, final Body body, final Local classLocal,
			final LocalGenerator gen) {
		final Set<SootClass> parentClasses = Collections.<SootClass> emptySet();
		return buildMethodCall(methodToCall, body, classLocal, gen, parentClasses);
	}

	/**
	 * Create invoke statement to call a method
	 *
	 * @param methodToCall
	 *            - method to be invoked
	 * @param body
	 *            - dummy main method body
	 * @param classLocal
	 *            - class variable
	 * @param gen
	 *            - soot local generator object
	 *
	 * @param parentClasses
	 *            - parent classes of method's class
	 * @return Soot Stmt
	 */

	private Stmt buildMethodCall(final SootMethod methodToCall, final Body body, final Local classLocal,
			final LocalGenerator gen, final Set<SootClass> parentClasses) {

		assert methodToCall != null : "Current method was null";
		assert body != null : "Body was null";
		assert gen != null : "Local generator was null";

		if (classLocal == null && !methodToCall.isStatic()) {
			this.failedMethods.add(methodToCall);
			return null;
		}

		final InvokeExpr invokeExpr;
		final List<Value> args = new LinkedList<>();

		/* Create invoke expression for Methods that take parameters */
		if (methodToCall.getParameterCount() > 0) {
			for (final Type tp : methodToCall.getParameterTypes()) {
				args.add(getValueForType(body, gen, tp, new HashSet<SootClass>(), parentClasses));
			}

			/*
			 * Check of method is static, if yes create a static invoke
			 * expression
			 */

			if (methodToCall.isStatic()) {
				invokeExpr = Jimple.v().newStaticInvokeExpr(methodToCall.makeRef(), args);
			} else {
				assert classLocal != null : "Class local method was null for non-static method call";
				if (methodToCall.isConstructor()) {
					invokeExpr = Jimple.v().newSpecialInvokeExpr(classLocal, methodToCall.makeRef(), args);
				} else {
					invokeExpr = Jimple.v().newVirtualInvokeExpr(classLocal, methodToCall.makeRef(), args);
				}
			}

			/* Create invoke expression for methods that take no parameter */
		} else {

			/*
			 * Check of method is static, if yes create a static invoke
			 * expression
			 */
			if (methodToCall.isStatic()) {
				invokeExpr = Jimple.v().newStaticInvokeExpr(methodToCall.makeRef());
			} else {
				assert classLocal != null : "Class local method was null for non-static method call";
				if (methodToCall.isConstructor()) {
					invokeExpr = Jimple.v().newSpecialInvokeExpr(classLocal, methodToCall.makeRef());
				} else {
					invokeExpr = Jimple.v().newVirtualInvokeExpr(classLocal, methodToCall.makeRef());
				}
			}
		}

		Stmt stmt;

		/*
		 * If method returns a value, create local variable and assign the
		 * return value from the method
		 */
		if (!(methodToCall.getReturnType() instanceof VoidType)) {
			final Local returnLocal = gen.generateLocal(methodToCall.getReturnType());
			stmt = Jimple.v().newAssignStmt(returnLocal, invokeExpr);

		} else {
			stmt = Jimple.v().newInvokeStmt(invokeExpr);
		}
		body.getUnits().add(stmt);

		return stmt;
	}

	/**
	 *
	 * @param body
	 * @param gen
	 * @param type
	 * @param constructionStack
	 * @param parentClasses
	 * @return
	 */
	private Value getValueForType(final Body body, final LocalGenerator gen, final Type type,
			final Set<SootClass> constructionStack, final Set<SootClass> parentClasses) {
		/*
		 * Depending on the final parameter type, we try final to find a final
		 * suitable concrete substitution
		 */
		if (isSimpleType(type.toString())) {
			return getSimpleDefaultValue(type.toString());
		} else if (type instanceof RefType) {
			final SootClass classToType = ((RefType) type).getSootClass();

			if (classToType != null) {
				/*
				 * If we have a parent class compatible with this type, we use
				 * it before we check any other option
				 */
				for (final SootClass parent : parentClasses) {
					if (isCompatible(parent, classToType)) {
						final Value val = this.localVarClass.get(parent.getName());
						if (val != null) {
							return val;
						}
					}
				}

				final Value val = generateClassConstructor(classToType, body, constructionStack, parentClasses);

				/* If we cannot create a parameter, we try a null reference. */

				if (val == null) {
					return NullConstant.v();
				}

				return val;
			}
		} else if (type instanceof ArrayType) {
			final Value arrVal = buildArrayOfType(body, gen, (ArrayType) type, constructionStack, parentClasses);
			if (arrVal == null) {
				return NullConstant.v();
			}
			return arrVal;
		} else {
			return null;
		}
		throw new RuntimeException("Found no value for type");
	}

	/**
	 * Check if one class is compatible with the another
	 *
	 * @param actual
	 *            class one
	 * @param expected
	 *            class two
	 * @return true if class one is compatible with two, false otherwise
	 */
	private boolean isCompatible(final SootClass actual, final SootClass expected) {
		SootClass act = actual;
		while (true) {

			if (act.getName().equals(expected.getName())) {
				return true;
			}

			/*
			 * If we expect an interface, the current class might implement it
			 */
			if (expected.isInterface()) {
				for (final SootClass intf : act.getInterfaces()) {
					if (intf.getName().equals(expected.getName())) {
						return true;
					}
				}
			}

			/*
			 * If we cannot continue our search further up the hierarchy, the
			 * two types are incompatible
			 */
			if (!act.hasSuperclass()) {
				return false;
			}
			act = act.getSuperclass();
		}
	}

	/**
	 * Create array local for a given type
	 *
	 * @param body
	 *            - dummy main method body
	 * @param gen
	 *            - Soot Local Generator object
	 * @param type
	 *            - type
	 * @param constructionStack
	 *            - construction stack set of classes
	 * @param parentClasses
	 *            - list of parent classes
	 * @return - local variable for the array type
	 */
	private Value buildArrayOfType(final Body body, final LocalGenerator gen, final ArrayType type,
			final Set<SootClass> constructionStack, final Set<SootClass> parentClasses) {
		final Local local = gen.generateLocal(type);

		/* Generate a new single-element array */
		final NewArrayExpr newArrayExpr = Jimple.v().newNewArrayExpr(type.getElementType(), IntConstant.v(1));
		final AssignStmt assignArray = Jimple.v().newAssignStmt(local, newArrayExpr);
		body.getUnits().add(assignArray);

		/* Generate a single element in the array */
		final AssignStmt assign = Jimple.v().newAssignStmt(Jimple.v().newArrayRef(local, IntConstant.v(0)),
				getValueForType(body, gen, type.getElementType(), constructionStack, parentClasses));
		body.getUnits().add(assign);
		return local;
	}

	/**
	 * Build method call for a given method name
	 *
	 * @param subsignature
	 *            method name
	 * @param currentClass
	 *            Class of the method
	 * @param entryPoints
	 *            entry points for the android application
	 * @param classLocal
	 *            class variable
	 * @return Soot Stmt object
	 */
	private Stmt searchAndBuildMethod(final String subsignature, final SootClass currentClass,
			final Set<String> entryPoints, final Local classLocal) {
		if (currentClass == null || classLocal == null) {
			return null;
		}

		final SootMethod method = findMethod(currentClass, subsignature);
		if (method == null) {

			return null;
		}
		entryPoints.remove(method.getSignature());

		/* If method is Android library method, we will not call the method */
		if (method.getDeclaringClass().getName().startsWith("android.")) {
			return null;
		}

		assert method.isStatic() || classLocal != null : "Class local was null for non-static method "
				+ method.getSignature();

		/* Build invoke statement for the method call */
		return buildMethodCall(method, this.body, classLocal, this.generator);
	}

	/**
	 * Model the life cycle for Activity classes
	 *
	 * @param entryPoints
	 *            Entry points of the Android application
	 * @param currentClass
	 *            Android Activity class
	 * @param classLoc
	 *            class variable of the Activity class
	 */
	private void generateActivityLifecycle(final Set<String> entryPoints, final SootClass currentClass,
			final Local classLoc) {

		final Set<SootClass> referenceClasses = new HashSet<>();
		if (this.applicationClass != null) {
			referenceClasses.add(this.applicationClass);
		}
		for (final SootClass callbackClass : this.applCallbkClasses) {
			referenceClasses.add(callbackClass);
		}
		referenceClasses.add(currentClass);

		/* 1.Creating invoke statement for onCreate method */

		{

			searchAndBuildMethod(Constants.ACTIVITY_ONCREATE, currentClass, entryPoints, classLoc);
			addCallbackMethods(this.applicationClass, referenceClasses,
					Constants.APPLIFECYCLECALLBACK_ONACTIVITYCREATED);
		}

		/* 2.Creating invoke statement for onStart method */
		{

			searchAndBuildMethod(Constants.ACTIVITY_ONSTART, currentClass, entryPoints, classLoc);
			addCallbackMethods(this.applicationClass, referenceClasses,
					Constants.APPLIFECYCLECALLBACK_ONACTIVITYSTARTED);

			searchAndBuildMethod(Constants.ACTIVITY_ONRESTOREINSTANCESTATE, currentClass, entryPoints, classLoc);
			searchAndBuildMethod(Constants.ACTIVITY_ONPOSTCREATE, currentClass, entryPoints, classLoc);

		}

		/* 3.Creating invoke statement for onResume method */

		{
			searchAndBuildMethod(Constants.ACTIVITY_ONRESUME, currentClass, entryPoints, classLoc);
			addCallbackMethods(this.applicationClass, referenceClasses,
					Constants.APPLIFECYCLECALLBACK_ONACTIVITYRESUMED);

			searchAndBuildMethod(Constants.ACTIVITY_ONPOSTRESUME, currentClass, entryPoints, classLoc);

		}

		/* Add callback methods for this Activity class */

		final boolean hasCallBacks = this.callbackFunctions.containsKey(currentClass.getName());
		if (hasCallBacks) {
			addCallbackMethods(currentClass);
		}

		/* 4. Creating invoke statement for onPause method */

		{
			searchAndBuildMethod(Constants.ACTIVITY_ONPAUSE, currentClass, this.entryPoints, classLoc);
			addCallbackMethods(this.applicationClass, referenceClasses,
					Constants.APPLIFECYCLECALLBACK_ONACTIVITYPAUSED);

			searchAndBuildMethod(Constants.ACTIVITY_ONCREATEDESCRIPTION, currentClass, this.entryPoints, classLoc);

			searchAndBuildMethod(Constants.ACTIVITY_ONSAVEINSTANCESTATE, currentClass, entryPoints, classLoc);
			addCallbackMethods(this.applicationClass, referenceClasses,
					Constants.APPLIFECYCLECALLBACK_ONACTIVITYSAVEINSTANCESTATE);
		}

		/* 5. Add invoke statement for onStop method */
		{
			searchAndBuildMethod(Constants.ACTIVITY_ONSTOP, currentClass, entryPoints, classLoc);
			addCallbackMethods(this.applicationClass, referenceClasses,
					Constants.APPLIFECYCLECALLBACK_ONACTIVITYSTOPPED);

		}

		/* 6.Add invoke statement for onRestart method */

		{
			searchAndBuildMethod(Constants.ACTIVITY_ONRESTART, currentClass, entryPoints, classLoc);

		}

		/* 7.Add invoke statement for onDestroy method */
		{
			searchAndBuildMethod(Constants.ACTIVITY_ONDESTROY, currentClass, entryPoints, classLoc);
			addCallbackMethods(this.applicationClass, referenceClasses,
					Constants.APPLIFECYCLECALLBACK_ONACTIVITYDESTROYED);
		}

	}

	/**
	 * Model Android Service life cycle
	 *
	 * @param entryPoints
	 *            Entry points of the Android application
	 * @param currentClass
	 *            Android Service class
	 * @param classLoc
	 *            class variable of the Service class
	 */
	private void generateServiceLifecycle(final Set<String> entryPoints, final SootClass currentClass,
			final Local classLoc) {

		final boolean isGCMBaseIntent = isGCMBaseIntentService(currentClass);
		final boolean isGCMListener = !isGCMBaseIntent && isGCMListenerService(currentClass);

		/* 1. Add invoke Statement for onCreate method */

		{

			searchAndBuildMethod(Constants.SERVICE_ONCREATE, currentClass, entryPoints, classLoc);
		}

		/* Services have two lifecycles */
		/* Lifecycle 1 */

		/* 2. Add invoke statment for onStart method */

		{
			searchAndBuildMethod(Constants.SERVICE_ONSTART1, currentClass, entryPoints, classLoc);

			searchAndBuildMethod(Constants.SERVICE_ONSTART2, currentClass, entryPoints, classLoc);

		}

		boolean hasAddMethods = false;
		if (isGCMBaseIntent) {
			for (final String signature : Constants.getGCMIntentServiceMethods()) {
				final SootMethod smethod = findMethod(currentClass, signature);
				if (smethod != null
						&& !smethod.getDeclaringClass().getName().equals(Constants.GCMBASEINTENTSERVICECLASS)) {
					hasAddMethods |= createPlainMethodCall(classLoc, smethod);
				}
			}
		} else if (isGCMListener) {
			for (final String signature : Constants.getGCMListenerServiceMethods()) {
				final SootMethod sMethod = findMethod(currentClass, signature);
				if (sMethod != null
						&& !sMethod.getDeclaringClass().getName().equals(Constants.GCMLISTENERSERVICECLASS)) {
					hasAddMethods |= createPlainMethodCall(classLoc, sMethod);
				}
			}
		}

		addCallbackMethods(currentClass);

		/* End of Lifecycle 1 */
		/* Lifecylce 2 */

		/* add Invoke statment for onBind method */
		{

			searchAndBuildMethod(Constants.SERVICE_ONBIND, currentClass, entryPoints, classLoc);
		}

		hasAddMethods = false;
		if (isGCMBaseIntent) {
			for (final String signature : Constants.getGCMIntentServiceMethods()) {
				final SootMethod smethod = findMethod(currentClass, signature);
				if (smethod != null
						&& !smethod.getDeclaringClass().getName().equals(Constants.GCMBASEINTENTSERVICECLASS)) {
					hasAddMethods |= createPlainMethodCall(classLoc, smethod);
				}
			}
		}
		addCallbackMethods(currentClass);

		/* add Invoke statment for onUnBind method */
		{

			searchAndBuildMethod(Constants.SERVICE_ONUNBIND, currentClass, entryPoints, classLoc);
		}

		/* add Invoke statment for onReBind method */
		{

			searchAndBuildMethod(Constants.SERVICE_ONREBIND, currentClass, entryPoints, classLoc);
		}

		/* Lifecycle 2 ends */

		/* add Invoke statment for onDestroy method */
		{

			searchAndBuildMethod(Constants.SERVICE_ONDESTROY, currentClass, entryPoints, classLoc);
		}

	}

	/**
	 * Check of class is GCMListenerService class
	 *
	 * @param currentClass
	 *            Service class
	 * @return true if GCMListenerService class, false otherwise
	 */
	private boolean isGCMListenerService(SootClass currentClass) {
		while (currentClass.hasSuperclass()) {
			if (currentClass.getSuperclass().getName().equals(Constants.GCMLISTENERSERVICECLASS)) {
				return true;
			}
			currentClass = currentClass.getSuperclass();
		}
		return false;
	}

	/**
	 * Check of class is GCMBaseIntentService class
	 *
	 * @param currentClass
	 *            Service class
	 * @return true if GCMBaseIntentService class, false otherwise
	 */
	private boolean isGCMBaseIntentService(SootClass currentClass) {

		while (currentClass.hasSuperclass()) {
			if (currentClass.getSuperclass().getName().equals(Constants.GCMBASEINTENTSERVICECLASS)) {
				return true;
			}
			currentClass = currentClass.getSuperclass();
		}
		return false;

	}

	/**
	 * Model Android Broadcast Receiver life cycle
	 *
	 * @param entryPoints
	 *            Entry points of the Android application
	 * @param currentClass
	 *            Android Broadcast Receiver class
	 * @param classLoc
	 *            class variable of the Broadcast Receiver class
	 */
	private void generateBroadcastReceiverLifecycle(final Set<String> entryPoints, final SootClass currentClass,
			final Local classLoc) {

		/* 1. Add invoke statement for onReceive method */
		{
			searchAndBuildMethod(Constants.BROADCAST_ONRECEIVE, currentClass, entryPoints, classLoc);
		}
		addCallbackMethods(currentClass);

	}

	/**
	 *
	 * Model Android Content Provider life cycle
	 *
	 * @param entryPoints
	 *            Entry points of the Android application
	 * @param currentClass
	 *            Android Content Provider class
	 * @param classLoc
	 *            class variable of the Content Provider class
	 */
	private void generateContentProviderLifecycle(final Set<String> entryPoints, final SootClass currentClass,
			final Local classLoc) {

		// onCreate method of content provider will be the first method to be
		// called. This has already been handled separately

		addCallbackMethods(currentClass);

	}

	/**
	 * Add call back methods present in the given Android component class
	 *
	 * @param currentClass
	 */
	private void addCallbackMethods(final SootClass currentClass) {
		addCallbackMethods(currentClass, null, "");
	}

	/**
	 * Add call back methods present in the given Android component class
	 *
	 * @param currentClass
	 *            Android component class
	 * @param referenceClasses
	 *            Set of reference classes
	 * @param callbackSignature
	 *            name of the call back method
	 */
	private void addCallbackMethods(final SootClass currentClass, Set<SootClass> referenceClasses,
			final String callbackSignature) {

		if (currentClass == null) {
			return;
		}

		/* Skip if class has no call back functions declared */
		if (!this.callbackFunctions.containsKey(currentClass.getName())) {
			return;
		}

		final Map<SootClass, Set<SootMethod>> callbackClasses = new HashMap<>();
		for (final SootMethod sMethod : this.callbackFunctions.get(currentClass.getName())) {

			if (!callbackSignature.isEmpty() && !callbackSignature.equals(sMethod.getSubSignature())) {
				continue;
			}

			final SootClass theClass = Scene.v().getSootClass(sMethod.getDeclaringClass().getName());
			final SootMethod theMethod = findMethod(theClass, sMethod.getSubSignature());
			if (theMethod == null) {
				continue;
			}

			/* Check if the method is one of the life cycle methods */

			if (getComponentType(theClass) == componentType.Activity
					&& Constants.getActivityLifecycleMethods().contains(theMethod.getSubSignature())) {
				continue;
			}
			if (getComponentType(theClass) == componentType.Service
					&& Constants.getServiceLifecycleMethods().contains(theMethod.getSubSignature())) {
				continue;
			}
			if (getComponentType(theClass) == componentType.BroadcastReceiver
					&& Constants.getBroadcastLifecycleMethods().contains(theMethod.getSubSignature())) {
				continue;
			}
			if (getComponentType(theClass) == componentType.ContentProvider
					&& Constants.getContentproviderLifecycleMethods().contains(theMethod.getSubSignature())) {
				continue;
			}

			if (callbackClasses.containsKey(theClass)) {
				callbackClasses.get(theClass).add(theMethod);
			} else {
				final Set<SootMethod> methods = new HashSet<>();
				methods.add(theMethod);
				callbackClasses.put(theClass, methods);
			}
		}

		if (referenceClasses == null || referenceClasses.isEmpty()) {
			referenceClasses = Collections.singleton(currentClass);
		} else {
			referenceClasses = new HashSet<SootClass>(referenceClasses);
			referenceClasses.add(currentClass);
		}

		for (final SootClass callbackClass : callbackClasses.keySet()) {
			/*
			 * If we already have a parent class that defines this callback, we
			 * use it. Otherwise, we create a new one.
			 */
			final Set<Local> classLocals = new HashSet<>();
			for (final SootClass parentClass : referenceClasses) {
				final Local parentLocal = this.localVarClass.get(parentClass.getName());
				if (isCompatible(parentClass, callbackClass)) {
					classLocals.add(parentLocal);
				}
			}
			if (classLocals.isEmpty()) {
				/*
				 * Create a new instance of this class if we need to call a
				 * constructor, we insert the respective Jimple statement here
				 */
				final Local classLocal = generateClassConstructor(callbackClass, this.body, referenceClasses);
				if (classLocal == null) {
					LOGGER.warn("Constructor cannot be generated for callback class {}", callbackClass.getName());
					continue;
				}
				classLocals.add(classLocal);
			}

			/* Build the calls to all callback methods in this class */
			for (final Local classLocal : classLocals) {
				for (final SootMethod callbackMethod : callbackClasses.get(callbackClass)) {
					buildMethodCall(callbackMethod, this.body, classLocal, this.generator, referenceClasses);
				}
			}
		}

	}

	/**
	 * Create method calls for GCMBase intent and Listener classes
	 *
	 * @param classLocal
	 *            class variable
	 * @param currentMethod
	 *            method to be invoked
	 * @return false if method is service life cycle method, true otherwise
	 */
	private boolean createPlainMethodCall(final Local classLocal, final SootMethod currentMethod) {

		if (Constants.getServiceLifecycleMethods().contains(currentMethod.getSubSignature())) {
			return false;
		}
		buildMethodCall(currentMethod, this.body, classLocal, this.generator);
		return true;
	}

	/**
	 * Add call back methods from layout files to list of call back methods
	 *
	 * @param arscParser
	 */
	private void addlayoutCallBackMethods(final ARSCParser arscParser) {

		for (final Entry<String, List<Integer>> lcentry : this.actvtyLyoutMap.entrySet()) {
			final SootClass callbackClass = Scene.v().getSootClass(lcentry.getKey());
			int classId;
			final Iterator iterator = lcentry.getValue().iterator();
			while (iterator.hasNext()) {
				classId = (int) iterator.next();
				final AbstractResource resource = arscParser.findResource(classId);
				if (resource instanceof StringResource) {
					final String layoutFileName = ((StringResource) resource).getValue();
					final List<String> callbackMethods = this.layoutCallbackMap.get(layoutFileName);
					if (callbackMethods != null) {
						for (final String methodName : callbackMethods) {
							final String subSig = "void " + methodName + "(android.view.View)";
							SootClass currentClass = callbackClass;
							while (true) {
								final SootMethod callbackMethod = currentClass.getMethodUnsafe(subSig);
								if (callbackMethod != null) {
									this.callbackFunctions.get(callbackClass.getName()).add(callbackMethod);
									break;
								}
								if (!currentClass.hasSuperclass()) {

									break;
								}
								currentClass = currentClass.getSuperclass();
							}
						}
					}

				}
			}

		}

	}

}
