package de.upb.pga3.panda2.extension.lvl2a.graphgenerator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import soot.Body;
import soot.Local;
import soot.MethodOrMethodContext;
import soot.RefType;
import soot.Scene;
import soot.SceneTransformer;
import soot.SootClass;
import soot.SootMethod;
import soot.Type;
import soot.Unit;
import soot.Value;
import soot.jimple.DefinitionStmt;
import soot.jimple.IdentityStmt;
import soot.jimple.InstanceInvokeExpr;
import soot.jimple.IntConstant;
import soot.jimple.InvokeExpr;
import soot.jimple.ReturnVoidStmt;
import soot.jimple.Stmt;
import soot.jimple.toolkits.callgraph.ReachableMethods;
import soot.toolkits.graph.ExceptionalUnitGraph;
import soot.toolkits.scalar.SimpleLiveLocals;
import soot.toolkits.scalar.SmartLocalDefs;

/**
 *
 * @author RamKumar
 *
 *         The Implementation is based in the implementation in FlowDroid
 */

public final class CallBackAnalyser extends SceneTransformer {

	private final String sysClassAnd = "android.";
	private final String sysClassJava = "java.";
	private Set<String> androidCallBacks = new HashSet<>();
	private Set<String> entryPointClasses = new HashSet<>();
	private final Map<String, List<SootMethod>> callBackMethods = new HashMap<>();
	private final Map<String, List<Integer>> actvtyLayoutMap = new HashMap<>();
	public static CallBackAnalyser Instance = new CallBackAnalyser();

	/**
	 * Initialise two data structures that will be used in this class
	 *
	 * @param androidCallBacks
	 *            - List of android call back classes
	 * @param entryPointClasses
	 *            - entry points of android applications
	 */
	public void setInputs(final Set<String> androidCallBacks, final Set<String> entryPointClasses) {
		this.androidCallBacks = androidCallBacks;
		this.entryPointClasses = entryPointClasses;
	}

	/**
	 * Return single instance of this class
	 *
	 * @return instance of this class
	 */

	public static CallBackAnalyser getInstance() {
		return Instance;
	}

	/**
	 * Our own analysis for Soot.
	 */

	@Override
	protected void internalTransform(final String phaseName, final Map<String, String> options) {

		for (final String clName : this.entryPointClasses) {
			final SootClass sclass = Scene.v().getSootClass(clName);
			getLayoutIDs();
			final List<MethodOrMethodContext> methods = new ArrayList<>();
			methods.addAll(sclass.getMethods());
			analyseReachableMethods(sclass, methods);
			methodOverRideCallBackAnalysis(sclass);
		}

	}

	/**
	 * Find layout ID corresponding to an Activity class
	 */
	private void getLayoutIDs() {

		for (final String className : this.entryPointClasses) {
			final SootClass sclass = Scene.v().getSootClass(className);

			final Iterator<SootMethod> methodsIt = sclass.getMethods().iterator();
			// Scene.v().getReachableMethods().listener();
			while (methodsIt.hasNext()) {

				final SootMethod sMethod = methodsIt.next().method();

				final SootClass declaringClass = sMethod.getDeclaringClass();
				if (!sMethod.isConcrete()) {
					continue;
				}
				for (final Unit u : sMethod.retrieveActiveBody().getUnits()) {
					if (u instanceof Stmt) {
						final Stmt stmnt = (Stmt) u;
						if (stmnt.containsInvokeExpr()) {
							final InvokeExpr iExpr = stmnt.getInvokeExpr();

							if (isViewInvoked(iExpr)) {

								for (final Value v : iExpr.getArgs()) {
									if (v instanceof IntConstant) {
										final IntConstant intVal = (IntConstant) v;

										if (this.actvtyLayoutMap.containsKey(declaringClass.getName())) {
											this.actvtyLayoutMap.get(declaringClass.getName()).add(intVal.value);
										} else {

											final List<Integer> layoutIds = new ArrayList<>();
											layoutIds.add(intVal.value);
											this.actvtyLayoutMap.put(declaringClass.getName(), layoutIds);
										}
									}
								}
							}
						}
					}
				}

			}
		}

	}

	/**
	 * Check if the expression invokes one of Android's View classes.
	 *
	 * @param iExpr
	 *            invoke expression
	 *
	 * @return true if View class is invoked, false otherwise
	 */

	private boolean isViewInvoked(final InvokeExpr iExpr) {

		/* If the invoked method is setContentView(), View class is invoked */
		final SootMethod invkdMethod = iExpr.getMethod();
		if (invkdMethod.getName().equals("setContentView")) {
			return true;
		}

		SootClass curClass = iExpr.getMethod().getDeclaringClass();
		while (curClass != null) {
			if (curClass.getName().equals("android.app.Activity")
					|| curClass.getName().equals("android.support.v7.app.ActionBarActivity")) {
				return true;
			}
			if (curClass.declaresMethod("void setContentView(int)")) {
				return true;
			}
			curClass = curClass.hasSuperclass() ? curClass.getSuperclass() : null;
		}

		return false;
	}

	/**
	 * Check methods that override Android library method
	 *
	 * @param sclass
	 *            Class under analysis
	 */

	private void methodOverRideCallBackAnalysis(final SootClass sclass) {

		if (!sclass.isConcrete() || sclass.isInterface() || isSystemClass(sclass.getName())) {
			return;
		}

		/*
		 * If a user defined method overrides a method in Android OS class, we
		 * treat it as callback method. We obtain a list of methods in super
		 * classes of the current class
		 */

		final Set<String> systemMethods = new HashSet<>();
		for (final SootClass parentClass : Scene.v().getActiveHierarchy().getSuperclassesOf(sclass)) {
			if (isSystemClass(parentClass.getName())) {
				for (final SootMethod sm : parentClass.getMethods()) {
					if (!sm.isConstructor()) {
						systemMethods.add(sm.getSubSignature());
					}
				}
			}
		}

		/*
		 * Now we iterate through methods in the class to check if it has
		 * overridden any method from parent class
		 */

		for (final SootClass parentClass : Scene.v().getActiveHierarchy().getSubclassesOfIncluding(sclass)) {

			if (parentClass.getName().startsWith(this.sysClassAnd)) {
				continue;
			}
			for (final SootMethod method : parentClass.getMethods()) {
				if (!systemMethods.contains(method.getSubSignature())) {
					continue;
				}
				/* This is a callback method */
				checkAndAddMethod(method, sclass);
			}
		}

	}

	/**
	 * Check if class is Android System or Java System class
	 *
	 * @param name
	 *            class name
	 * @return true if system class, false otherwise
	 */
	private boolean isSystemClass(final String name) {
		return name.startsWith(this.sysClassAnd) || name.startsWith(this.sysClassJava) || name.startsWith("sun.")
				|| name.startsWith("com.google.");

	}

	/**
	 * We analyse all methods reachable for call back registrations
	 *
	 * @param sclass
	 *            Class under analysis
	 * @param methods
	 *            List of methods in the class
	 */
	private void analyseReachableMethods(final SootClass sclass, final List<MethodOrMethodContext> methods) {

		final ReachableMethods recMeth = new ReachableMethods(Scene.v().getCallGraph(), methods);
		recMeth.update();
		final Iterator<MethodOrMethodContext> recMethIt = recMeth.listener();
		while (recMethIt.hasNext()) {
			final SootMethod smethod = recMethIt.next().method();
			callBackRegistrationAnalysis(sclass, smethod);
		}
	}

	/**
	 * Check for call back registrations
	 *
	 * @param lifecycleClass
	 *            Android component class
	 * @param smethod
	 *            Method under analysis
	 */
	private void callBackRegistrationAnalysis(final SootClass lifecycleClass, final SootMethod smethod) {

		/* The implementation of this method is similar to that in FlowDroid */

		/* We will not analyse methods belonging to System Classes */

		if (smethod.getDeclaringClass().getName().startsWith(this.sysClassAnd)
				|| smethod.getDeclaringClass().getName().startsWith(this.sysClassJava) || !smethod.isConcrete()) {
			return;
		}

		final ExceptionalUnitGraph graph = new ExceptionalUnitGraph(smethod.retrieveActiveBody());
		final SmartLocalDefs smd = new SmartLocalDefs(graph, new SimpleLiveLocals(graph));

		/* Iterate over all statement and find callback registration methods */
		final Set<SootClass> callbackClasses = new HashSet<>();
		for (final Unit u : smethod.retrieveActiveBody().getUnits()) {
			final Stmt stmt = (Stmt) u;

			/*
			 * Callback registrations are always instance invoke expressions
			 */

			if (stmt.containsInvokeExpr() && stmt.getInvokeExpr() instanceof InstanceInvokeExpr) {
				final InstanceInvokeExpr iinv = (InstanceInvokeExpr) stmt.getInvokeExpr();

				final String[] parameters = parseParmaetersFromSubSignature(
						iinv.getMethodRef().getSubSignature().getString());
				for (int i = 0; i < parameters.length; i++) {
					final String param = parameters[i];
					if (this.androidCallBacks.contains(param)) {
						final Value arg = iinv.getArg(i);

						/*
						 * If we have a formal parameter type that corresponds
						 * to one of the Android callback interfaces, we look
						 * for definitions of the parameter to estimate the
						 * actual type.
						 */
						if (arg.getType() instanceof RefType && arg instanceof Local) {
							for (final Unit def : smd.getDefsOfAt((Local) arg, u)) {
								assert def instanceof DefinitionStmt;
								final Type type = ((DefinitionStmt) def).getRightOp().getType();
								if (type instanceof RefType) {
									final SootClass callbackClass = ((RefType) type).getSootClass();
									if (callbackClass.isInterface()) {
										for (final SootClass impl : Scene.v().getActiveHierarchy()
												.getImplementersOf(callbackClass)) {
											for (final SootClass c : Scene.v().getActiveHierarchy()
													.getSubclassesOfIncluding(impl)) {
												callbackClasses.add(c);
											}
										}
									} else {
										for (final SootClass c : Scene.v().getActiveHierarchy()
												.getSubclassesOfIncluding(callbackClass)) {
											callbackClasses.add(c);
										}
									}
								}
							}
						}
					}
				}
			}
		}

		for (final SootClass sClass : callbackClasses) {
			analyzeSootClass(sClass, lifecycleClass);
		}
	}

	/**
	 * Analyse Call back class
	 *
	 * @param sClass
	 *            call back class
	 * @param lifecycleClass
	 *            Android life cycle class
	 */
	private void analyzeSootClass(final SootClass sClass, final SootClass lifecycleClass) {

		/* We will not analyse System classes */
		if (sClass.getName().startsWith(this.sysClassAnd) || sClass.getName().startsWith(this.sysClassJava)) {
			return;
		}

		/* Check of callbacks are implemented via interfaces */

		checkForCallBackInterfaces(sClass, sClass, lifecycleClass);

	}

	/**
	 * Check recursively all interfaces of the call back class
	 *
	 * @param baseClass
	 *            Call back class
	 * @param sClass
	 *            Super class
	 * @param lifecycleClass
	 *            Android lifecycle class
	 */
	private void checkForCallBackInterfaces(final SootClass baseClass, final SootClass sClass,
			final SootClass lifecycleClass) {

		/*
		 * For abstract classes, we will not look for interface implementation
		 */

		/*
		 * For android package classes, interface implementation will not be
		 * checked
		 */
		if (!baseClass.isConcrete() || baseClass.getName().startsWith(this.sysClassAnd)) {
			return;
		}

		// If we are a class, one of our super classes might implement an
		// Android
		// interface

		if (sClass.hasSuperclass()) {
			checkForCallBackInterfaces(baseClass, sClass.getSuperclass(), lifecycleClass);
		}

		for (final SootClass i : collectAllInterfaces(sClass)) {

			if (this.androidCallBacks.contains(i.getName())) {

				for (final SootMethod sMethod : i.getMethods()) {
					checkAndAddMethod(getMethodFromHierarchyEx(baseClass, sMethod.getSubSignature()), lifecycleClass);
				}
			}
		}

	}

	/**
	 * Check and add method to list of call back methods
	 *
	 * @param sootmethod
	 *            method
	 * @param lifecycleClass
	 *            Android lifecycle class
	 */
	private void checkAndAddMethod(final SootMethod sootmethod, final SootClass lifecycleClass) {

		/* We skip android library methods */
		if (sootmethod.getDeclaringClass().getName().startsWith(this.sysClassAnd)
				|| sootmethod.getDeclaringClass().getName().startsWith(this.sysClassJava)) {
			return;
		}

		/* We skip concrete and empty methods */
		if (sootmethod.isConcrete() && isEmpty(sootmethod.retrieveActiveBody())) {
			return;
		}

		if (this.callBackMethods.containsKey(lifecycleClass.getName())) {
			this.callBackMethods.get(lifecycleClass.getName()).add(sootmethod);
		} else {
			final List<SootMethod> methods = new ArrayList<>();
			methods.add(sootmethod);
			this.callBackMethods.put(lifecycleClass.getName(), methods);
		}

	}

	/**
	 * Get SootMethod from a given class
	 *
	 * @param baseClass
	 *            class
	 * @param subSignature
	 *            name of method
	 * @return SootMethod object of method
	 */

	private SootMethod getMethodFromHierarchyEx(final SootClass baseClass, final String subSignature) {

		if (baseClass.declaresMethod(subSignature)) {
			return baseClass.getMethod(subSignature);
		}
		if (baseClass.hasSuperclass()) {
			return getMethodFromHierarchyEx(baseClass.getSuperclass(), subSignature);
		}
		throw new RuntimeException("Could not find method");
	}

	/**
	 * Collect all the interfaces for a class
	 *
	 * @param sClass
	 *            class
	 * @return list of Interface classes
	 */

	private Set<SootClass> collectAllInterfaces(final SootClass sClass) {
		final Set<SootClass> interfaces = new HashSet<>(sClass.getInterfaces());
		for (final SootClass i : sClass.getInterfaces()) {
			interfaces.addAll(collectAllInterfaces(i));
		}
		return interfaces;
	}

	/**
	 * Obtain the parameters from the method's sub signature
	 *
	 * @param subSignature
	 *            method name
	 * @return list of parameter names
	 */

	private String[] parseParmaetersFromSubSignature(final String subSignature) {
		Pattern pattSubsigToName = null;
		if (pattSubsigToName == null) {
			final Pattern pattern = Pattern.compile("^\\s*(.+)\\s+(.+)\\((.*?)\\)\\s*$");
			pattSubsigToName = pattern;
		}
		final Matcher matcher = pattSubsigToName.matcher(subSignature);
		if (!matcher.find() || matcher.groupCount() < 3) {
			return null;
		}

		final String params = matcher.group(3);
		return params.split("\\s*,\\s*");
	}

	/**
	 * Check if the body of a method is empty
	 *
	 * @param activeBody
	 * @return true if empty, false otherwise
	 */
	private boolean isEmpty(final Body activeBody) {
		for (final Unit u : activeBody.getUnits()) {
			if (!(u instanceof IdentityStmt || u instanceof ReturnVoidStmt)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Returns call back method map
	 *
	 * @return call back method map
	 */
	public Map<String, List<SootMethod>> getCallBackMethods() {
		return this.callBackMethods;
	}

	/**
	 * Returns activity layout id map
	 *
	 * @return activity layout id map
	 */
	public Map<String, List<Integer>> getActivityLayoutIDMap() {
		return this.actvtyLayoutMap;
	}

}
