/**
 *
 */
package de.upb.pga3.panda2.extension.lvl2a;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.datastructures.DetailLevel;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisResultLvl2a.DetailLevelLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ClassElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.FlowPath;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.MethodElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.PathElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ResourceElement;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.StatementElement;

/**
 * HTML table builder is used to build a table in HTML.
 *
 * @author nptsy
 */
public class HtmlTableBuilder {

	// logger
	private static final Logger LOGGER = LogManager.getLogger(HtmlTableBuilder.class);
	// main content
	private StringBuilder mTableComponent;
	// information header
	private StringBuilder mInfoContent;
	// number of columns: summary-2 columns and comparison-3 columns
	private int mIntCols;
	// highlight row
	private boolean mBHighLineRow = false;
	// is comparison mode
	private boolean mBComparisionMode = false;
	// // number of new paths
	// private int mINumberOfNewPath = 0;
	// // number of removed paths
	// private int mINumberOfRemovedPath = 0;

	public static String cssStyle = "body { background: #F4F4F4; font-family: Verdana, Geneva, sans-serif; font-size: 12px;}"
			+

	"#level2a { width:100%; border-collapse:collapse; }" +

	"#level2a td { border: 1px solid #000000; padding: 3px 7px 2px 7px; vertical-align:text-top;}" +

	"#level2a tr.alt td	{ color: #000000; background-color: #eeeeee;}" +

	"#level2a th {background-color:gray; color: black; border:1px solid #000000; font-size: 15px; text-align: left; padding:3px 7px 2px 7px; padding-top:5px; padding-bottom:4px; }"
			+

	"#level2a th.alt {background-color:#AAAAAA; color:black; font-size:13px; text-align: left; padding:3px 7px 2px 7px; padding-top:5px; padding-bottom:4px; }"
			+

	"#level2a tr.alt th {background-color: #AAAAAA; color: black; font-size: 13px; text-align: left; padding: 3px 7px 2px 7px; padding-top: 5px; padding-bottom: 4px; }"
			+

	"div.legend {background: #FFFFFF; border: 1px inset #CCCCCC; padding: 10px 10px 10px 10px; position: fixed; top: 10px; right: 10px;}"
			+

	"div.icon { float: left; border: 1px solid #000000; width: 12px; height: 7px; margin-right: 5px; margin-top: 3px;}"
			+

	"div.stats {display: inline-block; background: #FFFFFF; border: 1px inset #CCCCCC; padding:10px 10px 10px 10px; width: auto; height: auto;}"
			+

	".list-permission li { position:relative;list-style-type:none;text-indent:-2em;} " +

	".list-permission li:before { content: ' ';}" +

	".list-class li { position: relative; list-style-type: none; text-indent: -2em;}" +

	".list-method li { position: relative; list-style-type: none; text-indent: -2em;}" +

	".list-statement li { position: relative; list-style-type: none; text-indent: -2em;}" +

	".list-class li:before { content: '\\27A4';}" +

	".list-method li:before { content:'\\00BB';}" +

	".list-statement li:before { content:'\\25BA'; } ";

	public static String strLegend = "<div class=\"legend\"><center><strong>Legend</strong></center><br /><div class=\"icon\" style=\"background:#ff0000;\"></div>SOURCE PERMISSION<br /><div class=\"icon\" style=\"background:#0006fd;\"></div>SINK PERMISSION<br /></div>";

	/**
	 * Constructor. This is a constructor for comparison mode
	 *
	 * @param inNumCol
	 * @param inIsComparisionMode
	 */
	public HtmlTableBuilder(final boolean inIsComparisionMode, final String inAppName) {

		// initialize the table component
		this.mTableComponent = new StringBuilder("");

		// initialize the info content
		this.mInfoContent = new StringBuilder("");

		if (inIsComparisionMode) {
			// number of column
			this.mIntCols = 3;
		} else {
			this.mIntCols = 2;
		}

		// comparison mode?
		this.mBComparisionMode = inIsComparisionMode;

		// initialize table
		initTable(inAppName);
	}

	/**
	 * process result according to specific detail level in SUMMARY mode
	 *
	 * @param inPaths
	 *            the flow path
	 * @param inDetailLvl
	 *            the detail level
	 */
	public void processResult(final Collection<FlowPath> inPaths, final DetailLevel inDetailLvl) {

		addAnalysisInfo(false, inDetailLvl);

		// in case the list of filters has no source and no sink
		if (inDetailLvl.equals(DetailLevelLvl2a.RES_TO_RES)) {
			/*
			 * process resource to resource
			 */
			computeResourceSummary(inPaths);
		} else {
			if (inDetailLvl.equals(DetailLevelLvl2a.COMPONENT)) {
				/*
				 * process component
				 */
				computeComponentSummary(inPaths);
			} else {
				if (inDetailLvl.equals(DetailLevelLvl2a.STATEMENT)) {
					/*
					 * process statement flow
					 */
					computeStatementSummary(inPaths);
				} else {
					if (inDetailLvl.equals(DetailLevelLvl2a.METHOD)) {
						/*
						 * process method flow
						 */
						computeMethodSummary(inPaths);
					} else {
						// write log
						LOGGER.warn("Unknown analysis detail level is specified.");
					}
				}
			}
		}
	}

	/**
	 * process result according to specific detail level in SUMMARY mode
	 *
	 * @param inNewPaths
	 *            the new flow path
	 * @param inRemovePaths
	 *            the removed flow path
	 * @param inDetailLvl
	 *            the specific detail level
	 */
	public void processResult(final Collection<FlowPath> inNewPaths, final Collection<FlowPath> inRemovePaths,
			final DetailLevel inDetailLvl) {

		addAnalysisInfo(true, inDetailLvl);

		// in case the list of filters has no source and no sink
		if (inDetailLvl.equals(DetailLevelLvl2a.RES_TO_RES)) {
			/*
			 * process resource to resource
			 */
			computeResourceComparison(inNewPaths, inRemovePaths);
		} else {
			if (inDetailLvl.equals(DetailLevelLvl2a.COMPONENT)) {
				/*
				 * process component
				 */
				computeComponentComparison(inNewPaths, inRemovePaths);
			} else {
				if (inDetailLvl.equals(DetailLevelLvl2a.STATEMENT)) {
					/*
					 * process statement flow
					 */
					computeStatementComparison(inNewPaths, inRemovePaths);
				} else {
					if (inDetailLvl.equals(DetailLevelLvl2a.METHOD)) {
						/*
						 * process method flow
						 */
						computeMethodComparison(inNewPaths, inRemovePaths);
					} else {
						// write log
						LOGGER.warn("Unknown analysis detail level is specified.");
					}
				}
			}
		}
	}

	/**
	 * Constructor. By default, this is the constructor for summary mode
	 *
	 * @param inNumCol
	 */
	public HtmlTableBuilder(final int inNumCol, final String inAppName) {

		// initialize the table component
		this.mTableComponent = new StringBuilder("");

		// initialize the info content
		this.mInfoContent = new StringBuilder("");

		// number of column
		this.mIntCols = inNumCol;

		// initialize table
		initTable(inAppName);
	}

	/**
	 * add analysis information
	 *
	 * @param isComparisonMode
	 *            a flag for detecting COMPARISON mode or not
	 * @param inDetailLvl
	 *            the detail level
	 */
	public void addAnalysisInfo(final boolean isComparisonMode, final DetailLevel inDetailLvl) {
		if (!isComparisonMode) {
			this.mInfoContent.append("<strong>Analysis Mode: </strong>Summary<br/>\n");
		} else {
			this.mInfoContent.append("<strong>Analysis Mode: </strong>Comparision<br/>\n");
		}
		this.mInfoContent.append("<strong>Detail Level: </strong>" + inDetailLvl.toString() + "<br/>\n");
	}

	/**
	 * add row for comparison mode in detail level COMPONENT
	 *
	 * @param inNewLstPaths
	 *            the list of new flow paths in newer application
	 * @param inNewInput
	 *            the enhancedInput of the new application
	 * @param inRemovedLstPaths
	 *            the list of removed paths in old application
	 * @param inOldInput
	 *            the enhancedInput of the old application
	 * @param inLstFilters
	 *            the list of filtered permission
	 */
	public void computeComponentComparison(final Collection<FlowPath> inNewLstPaths,
			final Collection<FlowPath> inRemovedLstPaths) {

		// invalid paths
		if ((inNewLstPaths == null || inNewLstPaths.isEmpty())
				&& (inRemovedLstPaths == null || inRemovedLstPaths.isEmpty())) {
			LOGGER.warn("List of found paths is invalid ");
			return;
		}

		final Map<String, List<String>> mapNewPaths = getComponentComparison(true, inNewLstPaths);
		final Map<String, List<String>> mapRemovedPaths = getComponentComparison(false, inRemovedLstPaths);

		addRowForComparisonVer2(mapNewPaths, mapRemovedPaths);
	}

	/**
	 * add row for components into the current table
	 *
	 * @param inPaths
	 *            list of flow paths
	 * @param inInput
	 *            the EnhancedInput. This is used for getting component
	 *            corresponding to the statement
	 * @param inLstFilters
	 *            list of filters
	 */
	private void computeComponentSummary(final Collection<FlowPath> inPaths) {
		if (inPaths != null && !inPaths.isEmpty()) {

			final Map<String, List<String>> mapResults = new HashMap<>();

			for (final FlowPath inFlowPath : inPaths) {

				/*
				 * ---- get source and sink permission ---
				 */
				final ResourceElement soPerm = inFlowPath.getSource();

				final ResourceElement siPerm = inFlowPath.getSink();

				final StringBuilder row = new StringBuilder("");
				final StringBuilder pathContent = new StringBuilder("");
				pathContent.append("<ul class=\"list-class\">");

				// source permission
				final String sourcePerm = setColor(soPerm.getPermissionName(), "red");

				/*
				 * THE FIRST COLUMN
				 */
				row.append(generateCol(sourcePerm));

				// sink permission
				final String sinkPerm = setColor(siPerm.getPermissionName(), "blue");
				final String flowPath = processFlowPathsForComponents(inFlowPath);
				if (!flowPath.contains("<li>")) {
					continue;
				}

				// add content to the tag list ul
				pathContent.append(flowPath);
				// close tag
				pathContent.append("</ul>");

				pathContent.append("<ul class=\"list-permission\"><li>" + sinkPerm + "</li></ul>");
				final String col2 = generateCol(pathContent.toString());

				// connect 2 columns
				row.append(col2);

				List<String> lstValues = mapResults.get(soPerm.getPermissionName());
				if (lstValues == null) {
					lstValues = new ArrayList<>();
					lstValues.add(row.toString());
					mapResults.put(soPerm.getPermissionName(), lstValues);
				} else {
					if (!lstValues.contains(row.toString())) {
						lstValues.add(row.toString());
					}
				}
			}

			addRowForSummary(mapResults);
		}
	}

	/**
	 * add row for comparison mode- Detail level RESOURCE
	 *
	 * @param inNewLstPaths
	 * @param inRemovedLstPaths
	 * @param inLstFilters
	 */
	private void computeResourceComparison(final Collection<FlowPath> inNewLstPaths,
			final Collection<FlowPath> inRemovedLstPaths) {

		if (inNewLstPaths == null && inRemovedLstPaths == null) {
			LOGGER.warn("List of found paths is invalid ");
			return;
		}

		final Map<String, List<String>> mapNewPaths = new HashMap<>();
		final Map<String, List<String>> mapRemovedPaths = new HashMap<>();

		processFlowPathsForResource(inNewLstPaths, mapNewPaths);
		processFlowPathsForResource(inRemovedLstPaths, mapRemovedPaths);

		/*
		 * add 2 map of results into the HTML table
		 */
		final List<String> lstCheckedPermission = new ArrayList<>();
		final Set<String> setNewPermissions = mapNewPaths.keySet();
		for (final String keyPerm : setNewPermissions) {

			// this is the list of new paths in new version
			final List<String> newSiPermissions = mapNewPaths.get(keyPerm);

			// this is the list of removed paths in old version
			final List<String> oldSiPermission = mapRemovedPaths.get(keyPerm);

			final String source = setColor(keyPerm, "red");

			// add content to the second column
			for (final String newStrPerm : newSiPermissions) {
				// columns
				final StringBuilder column = new StringBuilder("");
				// open the list of flow
				final StringBuilder lstFlow = new StringBuilder("<ul>");
				final String sink = setColor(newStrPerm, "blue");
				// first column
				column.append(generateCol(source));
				// second column
				lstFlow.append(generateItem(sink));
				lstFlow.append("</ul>");
				column.append(generateCol(lstFlow.toString()));
				// third column
				column.append(generateCol(""));
				// add the row (combination of 3 columns to table)
				// add the string to the table
				final int iPost = this.mTableComponent.indexOf("</tbody>");
				this.mTableComponent.insert(iPost, generateRow(column.toString()));
				// this.mINumberOfNewPath++;
			}

			// add content to the third column
			if (oldSiPermission != null) {
				lstCheckedPermission.add(keyPerm);
				for (final String oldStrPerm : oldSiPermission) {
					final String sink = setColor(oldStrPerm, "blue");
					// columns
					final StringBuilder column = new StringBuilder("");
					// open the list of flow
					final StringBuilder lstFlow = new StringBuilder("<ul>");
					// first column
					column.append(generateCol(source));
					// second column
					column.append(generateCol(""));
					// third column
					lstFlow.append(generateItem(sink));
					lstFlow.append("</ul>");
					column.append(generateCol(lstFlow.toString()));

					// add the row (combination of 3 columns to table)
					// add the string to the table
					final int iPost = this.mTableComponent.indexOf("</tbody>");
					this.mTableComponent.insert(iPost, generateRow(column.toString()));
					// this.mINumberOfRemovedPath++;
				}
			}
		}

		final Set<String> setRemovedPermissions = mapRemovedPaths.keySet();
		for (final String keyPerm : setRemovedPermissions) {
			if (!lstCheckedPermission.contains(keyPerm)) {
				// this is the list of new paths in new version
				final List<String> removedPermissions = mapRemovedPaths.get(keyPerm);
				final String source = setColor(keyPerm, "red");
				// add content to the second column

				for (final String newStrPerm : removedPermissions) {
					final String sink = setColor(newStrPerm, "blue");
					// columns
					final StringBuilder column = new StringBuilder("");
					// open the list of flow
					final StringBuilder lstFlow = new StringBuilder("<ul>");
					// first column
					column.append(generateCol(source));
					// second column
					column.append(generateCol(""));
					// third column
					lstFlow.append(generateItem(sink));
					lstFlow.append("</ul>");
					column.append(generateCol(lstFlow.toString()));
					// add the row (combination of 3 columns to table)
					// add the string to the table
					final int iPost = this.mTableComponent.indexOf("</tbody>");
					this.mTableComponent.insert(iPost, generateRow(column.toString()));
					// this.mINumberOfRemovedPath++;
				}
			}
		}
	}

	/**
	 * add row for summary mode of detail level RESOURCE
	 *
	 * @param inPaths
	 *            list of flow paths
	 *
	 * @param inLstFilters
	 *            list of filters permission
	 */
	private void computeResourceSummary(final Collection<FlowPath> inPaths) {

		if (inPaths != null && !inPaths.isEmpty()) {

			/*
			 * process resource to resource
			 */
			final Map<ResourceElement, List<ResourceElement>> mapPerms = new HashMap<ResourceElement, List<ResourceElement>>();

			for (final FlowPath flow : inPaths) {
				final List<String> lstStrPerm = new ArrayList<String>();

				// source permission
				final ResourceElement sourcePerm = flow.getSource();
				final String strSourcePerm = sourcePerm.getPermissionName();

				// sink permission
				final ResourceElement sinkPerm = flow.getSink();
				final String strSinkPerm = sinkPerm.getPermissionName();

				final List<ResourceElement> lstPerms = mapPerms.get(sourcePerm);
				if (lstPerms == null) {
					final List<ResourceElement> newLstPerms = new ArrayList<>();
					newLstPerms.add(sinkPerm);
					mapPerms.put(sourcePerm, newLstPerms);
				} else {
					if (!lstPerms.contains(sinkPerm)) {
						lstPerms.add(sinkPerm);
					} else {
						continue;
					}
				}

				// process for showing only different sources and sinks
				/*
				 * NOTE: in the FlowPath object: the source and sink are always
				 * permission
				 */
				lstStrPerm.add(strSourcePerm);
				lstStrPerm.add(strSinkPerm);
				addRow(lstStrPerm);
			}
		}
	}

	/**
	 * add row for statement in Comparison mode. Detail-Level STATEMENT
	 *
	 * @param inNewLstPaths
	 *            the list of new flow paths of the newer application
	 * @param inRemovedLstPaths
	 *            the list of removed paths of the old application
	 * @param inNewInput
	 *            the enhancedInput of the new application
	 * @param inOldInput
	 *            the enhancedInput of the old application
	 * @param inLstFilters
	 *            the list of filtered permissions
	 */
	public void computeStatementComparison(final Collection<FlowPath> inNewLstPaths,
			final Collection<FlowPath> inRemovedLstPaths) {

		// invalid paths
		if ((inNewLstPaths == null || inNewLstPaths.isEmpty())
				&& (inRemovedLstPaths == null || inRemovedLstPaths.isEmpty())) {
			LOGGER.warn("List of found paths is invalid ");
			return;
		}

		final Map<String, List<String>> mapNewPaths = getStatementComparison(true, inNewLstPaths);
		final Map<String, List<String>> mapRemovedPaths = getStatementComparison(false, inRemovedLstPaths);

		// add content to the table
		addRowForComparisonVer2(mapNewPaths, mapRemovedPaths);
	}

	/**
	 * add rows for SUMMARY mode - detail level STATEMENT
	 *
	 * @param inPaths
	 *            the list of flow paths
	 * @param inInput
	 *            the enhancedInput
	 */
	public void computeStatementSummary(final Collection<FlowPath> inPaths) {

		// invalid flow path
		if (inPaths == null) {
			LOGGER.warn("Invalid flow path");
			return;
		}

		final Map<String, List<String>> mapPaths = new HashMap<>();

		for (final FlowPath inFlowPath : inPaths) {

			/*
			 * ---- get source and sink permission ---
			 */
			// source permission
			final ResourceElement sourcePerm = inFlowPath.getSource();
			// sink permission
			final ResourceElement sinkPerm = inFlowPath.getSink();

			final StringBuilder row = new StringBuilder("");
			final StringBuilder pathContent = new StringBuilder("");
			/*
			 * process a flow path
			 */

			final String strSourcePerm = setColor(sourcePerm.getPermissionName(), "red");

			/*
			 * THE FIRST COLUMN
			 */
			row.append(generateCol(strSourcePerm));

			final String strSinkPerm = setColor(sinkPerm.getPermissionName(), "blue");

			pathContent.append(processFlowPathsForStatements(inFlowPath));

			pathContent.append("<ul class=\"list-permission\"><li>" + strSinkPerm + "</li></ul>");
			final String col2 = generateCol(pathContent.toString());

			// connect 2 columns
			row.append(col2);

			List<String> lstValues = mapPaths.get(sourcePerm.getPermissionName());
			if (lstValues == null) {
				lstValues = new ArrayList<>();
				lstValues.add(row.toString());
				mapPaths.put(sourcePerm.getPermissionName(), lstValues);
			} else {
				if (!lstValues.contains(row.toString())) {
					lstValues.add(row.toString());
				}
			}
		}

		addRowForSummary(mapPaths);
	}

	/**
	 * collect result for detail level METHOD
	 *
	 * @param inNewLstPaths
	 *            the list of new flow paths
	 * @param inRemovedLstPaths
	 *            the list of removed flow paths
	 */
	private void computeMethodComparison(final Collection<FlowPath> inNewLstPaths,
			final Collection<FlowPath> inRemovedLstPaths) {

		// invalid paths
		if ((inNewLstPaths == null || inNewLstPaths.isEmpty())
				&& (inRemovedLstPaths == null || inRemovedLstPaths.isEmpty())) {
			LOGGER.warn("List of found paths is invalid ");
			return;
		}

		final Map<String, List<String>> mapNewPaths = getMethodComparison(true, inNewLstPaths);
		final Map<String, List<String>> mapRemovedPaths = getMethodComparison(false, inRemovedLstPaths);

		// add content to the table
		addRowForComparisonVer2(mapNewPaths, mapRemovedPaths);
	}

	/**
	 * collect result for method detail level
	 *
	 * @param inPaths
	 *            the flow path
	 */
	private void computeMethodSummary(final Collection<FlowPath> inPaths) {

		final Map<String, List<String>> mapPaths = new HashMap<>();

		for (final FlowPath inFlowPath : inPaths) {

			// length of path
			/*
			 * ---- get source and sink permission ---
			 */
			final ResourceElement soTran = inFlowPath.getSource();
			final ResourceElement siTran = inFlowPath.getSink();

			final StringBuilder row = new StringBuilder("");
			final StringBuilder pathContent = new StringBuilder("");
			/*
			 * process a flow path
			 */

			final String sourcePerm = setColor(soTran.getPermissionName(), "red");

			/*
			 * THE FIRST COLUMN
			 */
			row.append(generateCol(sourcePerm));

			final String sinkPerm = setColor(siTran.getPermissionName(), "blue");

			pathContent.append(processFlowPathsForMethods(inFlowPath));

			pathContent.append("<ul class=\"list-permission\"><li>" + sinkPerm + "</li></ul>");
			final String col2 = generateCol(pathContent.toString());

			// connect 2 columns
			row.append(col2);

			List<String> lstValues = mapPaths.get(soTran.getPermissionName());
			if (lstValues == null) {
				lstValues = new ArrayList<>();
				lstValues.add(row.toString());
				mapPaths.put(soTran.getPermissionName(), lstValues);
			} else {
				if (!lstValues.contains(row.toString())) {
					lstValues.add(row.toString());
				}
			}
		}

		addRowForSummary(mapPaths);
	}

	@Override
	public String toString() {
		String result = this.mInfoContent.toString();
		result += this.mTableComponent.toString();
		return result;
	}

	/**
	 * init the table for result
	 */
	private void initTable(final String inAppName) {

		// in case of summary mode
		if (!this.mBComparisionMode) {
			final String headerAppName = "App's name: " + inAppName;
			this.mTableComponent.append("<table style= \"width:100%\" id=\"level2a\"><tbody>");
			// header for app name
			this.mTableComponent.append("<tr><th colspan=\"2\">" + headerAppName + "	</th></tr>");
			// header for column name
			this.mTableComponent.append("<tr class=\"alt\"><th>Source(s)</th><th>Paths (to sinks)</th></tr>");
			// close tags
			this.mTableComponent.append("</tbody></table>");
		}
		// in case of comparison mode
		else {
			final String headerAppName = "App's name: " + inAppName;
			this.mTableComponent.append("<table style= \"width:100%\" id=\"level2a\"><tbody>");
			// header for app name
			this.mTableComponent.append("<tr><th colspan=\"3\">" + headerAppName + "	</th></tr>");
			// header for column name
			this.mTableComponent.append(
					"<tr class=\"alt\"><th>Source(s)</th><th>New paths in new version</th><th>Removed paths from current version</th></tr>");
			// close tags
			this.mTableComponent.append("</tbody></table>");
		}
	}

	/**
	 * process results for statements and transform them to html format -
	 * COMPARISON mode
	 *
	 * @param inIsNewPath
	 *            if the input list of FlowPath is new
	 * @param inLstFlowPaths
	 *            the input list of FlowPath
	 * @param inHavingSources
	 *            true if there exist source permissions source permission
	 *            otherwise false
	 * @param inHavingSinks
	 *            true if there exist sink permissions otherwise false
	 * @param inInput
	 *            the EnhancedInput
	 * @param inLstFilters
	 *            the list of filtered permission
	 * @return a map whose the key is a source permission, and value is a list
	 *         of entries that represents the flow path
	 */
	private Map<String, List<String>> getStatementComparison(final boolean inIsNewPath,
			final Collection<FlowPath> inLstFlowPaths) {

		final Map<String, List<String>> mapPaths = new HashMap<>();

		for (final FlowPath inFlowPath : inLstFlowPaths) {

			/*
			 * ---- get source and sink permission ---
			 */
			final ResourceElement soPerm = inFlowPath.getSource();
			final ResourceElement siPerm = inFlowPath.getSink();

			final StringBuilder row = new StringBuilder("");
			final StringBuilder pathContent = new StringBuilder("");
			/*
			 * process a flow path
			 */

			final String sourcePerm = setColor(soPerm.getPermissionName(), "red");

			/*
			 * THE FIRST COLUMN
			 */
			row.append(generateCol(sourcePerm));

			final String sinkPerm = setColor(siPerm.getPermissionName(), "blue");
			pathContent.append(processFlowPathsForStatements(inFlowPath));

			pathContent.append("<br/>" + "<ul class=\"list-permission\"><li>" + sinkPerm + "</li></ul>");
			final String col2 = generateCol(pathContent.toString());

			if (inIsNewPath) {
				// 2nd column
				row.append(col2);
				// 3rd column
				row.append(generateCol(""));
			} else {
				// 2nd column
				row.append(generateCol(""));
				// 3rd column
				row.append(col2);
			}

			List<String> lstValues = mapPaths.get(soPerm.getPermissionName());
			if (lstValues == null) {
				lstValues = new ArrayList<>();
				lstValues.add(row.toString());
				mapPaths.put(soPerm.getPermissionName(), lstValues);
			} else {
				if (!lstValues.contains(row.toString())) {
					lstValues.add(row.toString());
				}
			}
		}

		return mapPaths;
	}

	/**
	 * get information for (detail level) RESOURCE in the list of flow paths
	 *
	 * @param inLstFlowPaths
	 *            list of flow paths
	 * @param inLstFilters
	 *            list of filtered permissions
	 * @param hasSources
	 *            true if in the list of filters, sources exist otherwise false
	 * @param hasSinks
	 *            true if the the list of filters, sinks exist otherwise false
	 * @param mapResultPaths
	 *            the map that contains results
	 */
	private void processFlowPathsForResource(final Collection<FlowPath> inLstFlowPaths,
			final Map<String, List<String>> mapResultPaths) {
		for (final FlowPath flow : inLstFlowPaths) {
			// source permission
			final ResourceElement sourcePerm = flow.getSource();

			// sink permission
			final ResourceElement sinkPerm = flow.getSink();

			// add source and sink to map
			final List<String> lstStrPerms = mapResultPaths.get(sourcePerm.getPermissionName());
			if (lstStrPerms == null) {
				final List<String> newLstPerms = new ArrayList<>();
				newLstPerms.add(sinkPerm.getPermissionName());
				mapResultPaths.put(sourcePerm.getPermissionName(), newLstPerms);
			} else {
				if (!lstStrPerms.contains(sinkPerm.getPermissionName())) {
					lstStrPerms.add(sinkPerm.getPermissionName());
				} else {
					continue;
				}
			}
		}
	}

	/**
	 * process flow paths and pick up only android components to build html
	 * String
	 *
	 * @param inFlowPath
	 *            the input flow path
	 * @param inInput
	 *            the EnhancedInput
	 * @return an HTML String that contains the flow path
	 */
	private String processFlowPathsForComponents(final FlowPath inFlowPath) {
		String currentClass = "";
		final StringBuilder pathContent = new StringBuilder("");
		final int iLengthFlowPath = inFlowPath.getLength();
		for (int i = 1; i < iLengthFlowPath; i++) {
			if (i == iLengthFlowPath - 1) {
				break;
			}

			final PathElement pathEle = inFlowPath.getElement(i);

			if (pathEle instanceof ClassElement) {
				final ClassElement ce = (ClassElement) pathEle;
				if (ce.isAndroidComponent()) {
					// get name of the component
					final String strComp = ce.toFullString();
					if (strComp != null && !strComp.isEmpty()) {
						if (!currentClass.equals(strComp)) {
							currentClass = strComp;
							pathContent.append("<li> " + currentClass + "</li>");
						}
					}
				}
			}
		} // end loop of flow path

		return pathContent.toString();
	}

	/**
	 * process flow paths and pick up only statements to build html String
	 *
	 * @param inFlowPath
	 *            the flow path
	 * @param inInput
	 *            the enhancedInput
	 * @return an HTML string that contains the flow path
	 */
	private String processFlowPathsForStatements(final FlowPath inFlowPath) {
		final int iLengthFlowPath = inFlowPath.getLength();

		String currentClass = "";
		String currentMethod = "";
		final StringBuilder pathContent = new StringBuilder("<ul class=\"list-class\">");
		boolean tagClassOpen = false;
		boolean tagMethodOpen = false;
		boolean tagStatementOpent = false;
		PathElement pathEle;
		ClassElement cEle;
		MethodElement mEle;
		StatementElement sEle;

		for (int i = 1; i < iLengthFlowPath - 1; i++) {

			pathEle = inFlowPath.getElement(i);
			if (pathEle instanceof StatementElement) {
				sEle = (StatementElement) pathEle;
				mEle = sEle.getMethodElement();
				cEle = mEle.getClassElement();
				if (!cEle.getClassName().equals(currentClass)) {

					////////////////////////////////////////////
					// process closed tag first
					////////////////////////////////////////////
					if (tagStatementOpent) {
						// close item statement
						pathContent.append("</ul>");
						tagStatementOpent = false;

					}

					if (tagMethodOpen) {
						// close item method
						pathContent.append("</li>");
						pathContent.append("</ul>");
						tagMethodOpen = false;
					}

					if (tagClassOpen) {
						// close item class
						pathContent.append("</li>");
						tagClassOpen = false;
					}

					// CLASS
					pathContent.append("<li>" + cEle.getClassName());
					tagClassOpen = true;
					// METHOD
					pathContent.append("<ul class=\"list-method\">");
					pathContent.append("<li>" + encodeString(mEle.getMethodSubSignature()));
					tagMethodOpen = true;
					// STATEMENT
					pathContent.append("<ul class=\"list-statement\">");
					pathContent.append("<li>" + encodeString(sEle.toString()) + "</li>");
					tagStatementOpent = true;

					// -> Print class, method and statement here
					currentClass = cEle.getClassName();
					currentMethod = mEle.getMethodSubSignature();

				} else if (!mEle.getMethodSubSignature().equals(currentMethod)) {
					////////////////////////////////////////////
					// process closed tag first
					////////////////////////////////////////////
					if (tagStatementOpent) {
						// close item statement
						pathContent.append("</ul>");
						tagStatementOpent = false;

					}

					if (tagMethodOpen) {
						// close item method
						pathContent.append("</li>");
						pathContent.append("</ul>");
						tagMethodOpen = false;
					}
					// METHOD
					pathContent.append("<ul class=\"list-method\">");
					pathContent.append("<li>" + encodeString(mEle.getMethodSubSignature()));
					tagMethodOpen = true;
					// STATEMENT
					pathContent.append("<ul class=\"list-statement\">");
					pathContent.append("<li>" + encodeString(sEle.toString()) + "</li>");
					tagStatementOpent = true;

					// -> Print method and statement here
					currentMethod = mEle.getMethodSubSignature();
				} else {
					// -> Print statement here
					pathContent.append("<li>" + encodeString(sEle.toString()) + "</li>");
					tagStatementOpent = true;
				}
			}
		}

		////////////////////////////////////////////
		// process closed tag first
		////////////////////////////////////////////
		if (tagStatementOpent) {
			// close item statement
			pathContent.append("</ul>");
			tagStatementOpent = false;

		}

		if (tagMethodOpen) {
			// close item method
			pathContent.append("</li>");
			pathContent.append("</ul>");
			tagMethodOpen = false;
		}

		if (tagClassOpen) {
			// close item class
			pathContent.append("</li>");
			pathContent.append("</ul>");
			tagClassOpen = false;
		}

		return pathContent.toString();
	}

	/**
	 * process flow paths and pick up only methods of each statements to build
	 * HTML String
	 *
	 * @param inFlowPath
	 *            the flow path
	 * @param inInput
	 *            the enhancedInput
	 * @return an HTML string that contains the flow path
	 */
	private String processFlowPathsForMethods(final FlowPath inFlowPath) {
		final int iLengthFlowPath = inFlowPath.getLength();
		String currentClass = "";
		String currentMethod = "";
		final StringBuilder pathContent = new StringBuilder("<ul class=\"list-class\">");
		boolean tagClassOpen = false;
		boolean tagMethodOpen = false;
		PathElement pathEle;
		ClassElement cEle;
		MethodElement mEle;

		for (int i = 1; i < iLengthFlowPath - 1; i++) {

			pathEle = inFlowPath.getElement(i);
			if (pathEle instanceof MethodElement) {
				mEle = (MethodElement) pathEle;
				cEle = mEle.getClassElement();
				if (!cEle.getClassName().equals(currentClass)) {

					////////////////////////////////////////////
					// process closed tag first
					////////////////////////////////////////////

					if (tagMethodOpen) {
						// close item method
						pathContent.append("</li>");
						pathContent.append("</ul>");
						tagMethodOpen = false;
					}

					if (tagClassOpen) {
						// close item class
						pathContent.append("</li>");
						tagClassOpen = false;
					}

					// CLASS
					pathContent.append("<li>" + cEle.getClassName());
					tagClassOpen = true;
					// METHOD
					pathContent.append("<ul class=\"list-method\">");
					pathContent.append("<li>" + encodeString(mEle.getMethodSubSignature()));
					tagMethodOpen = true;

					// -> Print class, method and statement here
					currentClass = cEle.getClassName();
					currentMethod = mEle.getMethodSubSignature();

				} else if (!mEle.getMethodSubSignature().equals(currentMethod)) {
					////////////////////////////////////////////
					// process closed tag first
					////////////////////////////////////////////

					if (tagMethodOpen) {
						// close item method
						pathContent.append("</li>");
						pathContent.append("</ul>");
						tagMethodOpen = false;
					}
					// METHOD
					pathContent.append("<ul class=\"list-method\">");
					pathContent.append("<li>" + encodeString(mEle.getMethodSubSignature()));
					tagMethodOpen = true;

					// -> Print method and statement here
					currentMethod = mEle.getMethodSubSignature();
				}
			}
		}

		////////////////////////////////////////////
		// process closed tag first
		////////////////////////////////////////////

		if (tagMethodOpen) {
			// close item method
			pathContent.append("</li>");
			pathContent.append("</ul>");
			tagMethodOpen = false;
		}

		if (tagClassOpen) {
			// close item class
			pathContent.append("</li>");
			pathContent.append("</ul>");
			tagClassOpen = false;
		}

		return pathContent.toString();
	}

	/**
	 * add row to current table for SUMMARY mode. This is an official method for
	 * generating result
	 *
	 * @param inMapNewPaths
	 *            the list of flow paths
	 */
	private void addRowForSummary(final Map<String, List<String>> inMapNewPaths) {
		final Set<String> keyPerms = inMapNewPaths.keySet();
		for (final String perm : keyPerms) {
			final List<String> setEntries = inMapNewPaths.get(perm);
			for (final String strRow : setEntries) {
				// add the string to the table
				final int iPost = this.mTableComponent.indexOf("</tbody>");
				this.mTableComponent.insert(iPost, generateRow(strRow));
			}
		}
	}

	/**
	 * add row to table for COMPARISON mode. This method is official for
	 * generating html results debugging
	 *
	 * @param inMapNewPaths
	 *            the list of flow paths in the new version of an application
	 * @param inMapRemovedPaths
	 *            the list of flow paths in the old version of an application
	 */
	private void addRowForComparisonVer2(final Map<String, List<String>> inMapNewPaths,
			final Map<String, List<String>> inMapRemovedPaths) {

		final List<String> lstCheckedPermission = new ArrayList<>();
		final Set<String> keyPerms = inMapNewPaths.keySet();

		/*
		 * process for new path
		 */
		for (final String perm : keyPerms) {
			// new paths
			final List<String> lstNewEntries = inMapNewPaths.get(perm);
			// remove paths
			final List<String> lstRemovedEntries = inMapRemovedPaths.get(perm);

			for (final String newPath : lstNewEntries) {
				// add the string to the table
				final int iPost = this.mTableComponent.indexOf("</tbody>");
				this.mTableComponent.insert(iPost, generateRow(newPath));
				// this.mINumberOfNewPath++;
			}

			if (lstRemovedEntries != null) {
				// add to checked list
				lstCheckedPermission.add(perm);

				for (final String removedPath : lstRemovedEntries) {
					// add the string to the table
					final int iPost = this.mTableComponent.indexOf("</tbody>");
					this.mTableComponent.insert(iPost, generateRow(removedPath));
					// this.mINumberOfRemovedPath++;
				}
			}
		}

		/*
		 * process for removed path
		 */
		final Set<String> keyRemovedPerms = inMapRemovedPaths.keySet();
		for (final String removedPerm : keyRemovedPerms) {
			if (!lstCheckedPermission.contains(removedPerm)) {
				final List<String> lstRemovedPaths = inMapRemovedPaths.get(removedPerm);

				for (final String removedPath : lstRemovedPaths) {
					// add the string to the table
					final int iPost = this.mTableComponent.indexOf("</tbody>");
					this.mTableComponent.insert(iPost, generateRow(removedPath));
					// this.mINumberOfRemovedPath++;
				}
			}
		}
	}

	/**
	 * process results for components and transform them into HTML format -
	 * COMPARISON mode
	 *
	 * @param inIsNewPath
	 *            true if the added flow path is new
	 * @param inLstFlowPaths
	 *            the input list of FlowPaths
	 * @param inHavingSources
	 *            true if there exist source permissions otherwise false
	 * @param inHavingSinks
	 *            true if there exist sink permissions otherwise false
	 * @param inFilters
	 *            the list of filtered permissions
	 * @param inInput
	 *            the EnhancedInput
	 * @return a map whose the key is a source permission, and value is a list
	 *         of entries that represents the flow path
	 */
	private Map<String, List<String>> getComponentComparison(final boolean inIsNewPath,
			final Collection<FlowPath> inLstFlowPaths) {
		final Map<String, List<String>> mapResults = new HashMap<>();

		for (final FlowPath inFlowPath : inLstFlowPaths) {

			/*
			 * ---- get source and sink permission ---
			 */
			final ResourceElement soPerm = inFlowPath.getSource();
			final ResourceElement siPerm = inFlowPath.getSink();

			final StringBuilder row = new StringBuilder("");
			final StringBuilder pathContent = new StringBuilder("");
			pathContent.append("<ul class=\"list-class\">");

			// source permission
			final String sourcePerm = setColor(soPerm.getPermissionName(), "red");

			/*
			 * THE FIRST COLUMN
			 */
			row.append(generateCol(sourcePerm));

			// sink permission
			final String sinkPerm = setColor(siPerm.getPermissionName(), "blue");
			final String flowPath = processFlowPathsForComponents(inFlowPath);
			if (!flowPath.contains("<li>")) {
				continue;
			}
			// add content to the tag list ul
			pathContent.append(flowPath);
			// close tag
			pathContent.append("</ul>");
			// pathContent.append("<br/>" + "<ul
			// class=\"list-permission\"><li>"
			// + sinkPerm + "</li></ul>");
			pathContent.append("<ul class=\"list-permission\"><li>" + sinkPerm + "</li></ul>");
			final String col2 = generateCol(pathContent.toString());

			if (inIsNewPath) {
				// 2nd column
				row.append(col2);
				// 3rd column
				row.append(generateCol(""));
			} else {
				// 2nd column
				row.append(generateCol(""));
				// 3rd column
				row.append(col2);
			}

			List<String> lstValues = mapResults.get(soPerm.getPermissionName());
			if (lstValues == null) {
				lstValues = new ArrayList<>();
				lstValues.add(row.toString());
				mapResults.put(soPerm.getPermissionName(), lstValues);
			} else {
				if (!lstValues.contains(row.toString())) {
					lstValues.add(row.toString());
				}
			}
		}

		return mapResults;
	}

	/**
	 * set a specific color for a string. We have some basic color such as red,
	 * green, blue, yellow
	 *
	 * @param inContent
	 *            the string content that with be marked in color
	 * @param inColor
	 *            the color is used for marking the string content
	 * @return an HTML string
	 */
	private static String setColor(final String inContent, final String inColor) {
		final StringBuilder strBuilder = new StringBuilder("<font color=\"" + inColor + "\">" + inContent + "</font>");
		return strBuilder.toString();
	}

	/**
	 * generate an element li of list
	 *
	 * @param inContent
	 * @return
	 */
	private static String generateItem(final String inContent) {
		final StringBuilder strBuilder = new StringBuilder("<li>");
		strBuilder.append(inContent);
		strBuilder.append("</li>");
		return strBuilder.toString();
	}

	/**
	 * generate a column for a string content
	 *
	 * @param inContent
	 *            the string content
	 * @return a string with content embedded in html format
	 */
	private String generateCol(final String inContent) {
		final StringBuilder col = new StringBuilder("<td>");
		col.append(inContent);
		col.append("</td>");
		return col.toString();
	}

	/**
	 * generate a row for a string content
	 *
	 * @param inContent
	 *            the string content
	 * @return a string with content embedded in html format
	 */
	private String generateRow(final String inContent) {
		final StringBuilder row = new StringBuilder("");
		if (this.mBHighLineRow) {
			row.append("<tr class=\"alt\">");
			this.mBHighLineRow = false;
		} else {
			row.append("<tr>");
			this.mBHighLineRow = true;
		}
		row.append(inContent);
		row.append("</tr>");
		return row.toString();
	}

	/**
	 * encode special characters in a a string
	 *
	 * @param inContent
	 *            the content that will be embedded
	 * @return
	 */
	private static String encodeString(String inContent) {
		if (inContent != null) {
			inContent = inContent.replace("<", "&lt;");
			inContent = inContent.replace(">", "&gt;");
			return inContent;
		}
		return "";
	}

	/**
	 * add a row to current table
	 *
	 * @param inLstStrContent
	 */
	void addRow(final List<String> inLstStrContent) {
		if (inLstStrContent != null && !inLstStrContent.isEmpty()) {

			// columns
			final StringBuilder column = new StringBuilder("");
			// open the list of flow
			final StringBuilder lstFlow = new StringBuilder("<ul class= \"list-permission\">");

			final int minSize = Math.min(this.mIntCols, inLstStrContent.size());
			for (int i = 0; i < minSize; i++) {
				String inContent = inLstStrContent.get(i);

				// set color for source and sink
				if (i == 0) {
					// process for source here
					inContent = setColor(inContent, "red");
					// generate the first column here
					column.append(generateCol(inContent));
				} else {
					if (i == minSize - 1) {
						lstFlow.append(setColor(generateItem(inContent), "blue"));
					} else {
						lstFlow.append(generateItem(inContent));
					}
				}
			} // end getting list of content for column

			// close the list of flow
			lstFlow.append("</ul>");
			// generate the second column
			final String col2 = generateCol(lstFlow.toString());
			// connect 2 columns
			column.append(col2);

			// add the string to the table
			final int iPost = this.mTableComponent.indexOf("</tbody>");
			this.mTableComponent.insert(iPost, generateRow(column.toString()));
		}
	}

	/**
	 * process results for methods and transform them to HTML format -
	 * COMPARISON mode
	 *
	 * @param inIsNewPath
	 *            if the input list of FlowPath is new
	 * @param inLstFlowPaths
	 *            the input list of FlowPath
	 * @param inHavingSources
	 *            true if there exist source permissions source permission
	 *            otherwise false
	 * @param inHavingSinks
	 *            true if there exist sink permissions otherwise false
	 * @param inInput
	 *            the EnhancedInput
	 * @param inLstFilters
	 *            the list of filtered permission
	 * @return a map whose the key is a source permission, and value is a list
	 *         of entries that represents the flow path
	 */
	private Map<String, List<String>> getMethodComparison(final boolean inIsNewPath,
			final Collection<FlowPath> inLstFlowPaths) {

		final Map<String, List<String>> mapPaths = new HashMap<>();

		for (final FlowPath inFlowPath : inLstFlowPaths) {

			// length of path
			final int iLengthFlowPath = inFlowPath.getLength();

			/*
			 * ---- get source and sink permission ---
			 */
			final ResourceElement soPerm = inFlowPath.getSource();

			final ResourceElement siPerm = inFlowPath.getSink();

			final StringBuilder row = new StringBuilder("");
			final StringBuilder pathContent = new StringBuilder("");
			/*
			 * process a flow path
			 */

			final String sourcePerm = setColor(soPerm.getPermissionName(), "red");

			/*
			 * THE FIRST COLUMN
			 */
			row.append(generateCol(sourcePerm));

			final String sinkPerm = setColor(siPerm.getPermissionName(), "blue");
			pathContent.append(processFlowPathsForMethods(inFlowPath));

			pathContent.append("<br/>" + "<ul class=\"list-permission\"><li>" + sinkPerm + "</li></ul>");
			final String col2 = generateCol(pathContent.toString());

			if (inIsNewPath) {
				// 2nd column
				row.append(col2);
				// 3rd column
				row.append(generateCol(""));
			} else {
				// 2nd column
				row.append(generateCol(""));
				// 3rd column
				row.append(col2);
			}

			List<String> lstValues = mapPaths.get(soPerm.getPermissionName());
			if (lstValues == null) {
				lstValues = new ArrayList<>();
				lstValues.add(row.toString());
				mapPaths.put(soPerm.getPermissionName(), lstValues);
			} else {
				if (!lstValues.contains(row.toString())) {
					lstValues.add(row.toString());
				}
			}
		}

		return mapPaths;
	}
}
