package de.upb.pga3.panda2.extension.lvl2a;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Deque;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.mxgraph.canvas.mxSvgCanvas;
import com.mxgraph.layout.mxGraphLayout;
import com.mxgraph.layout.mxOrganicLayout;
import com.mxgraph.util.mxPoint;
import com.mxgraph.util.mxRectangle;
import com.mxgraph.util.mxXmlUtils;
import com.mxgraph.view.mxGraph;

/**
 *
 * @author Fabian
 *
 */
public class GroupedLayout extends mxGraphLayout {

	private static final double BORDER = 20.0;

	private final mxOrganicLayout layout;

	public GroupedLayout(final mxGraphLayout layout) {

		super(layout.getGraph());
		this.layout = (mxOrganicLayout) layout;
	}

	@Override
	public void execute(final Object parentVertex) {

		final Deque<Object> stack = new LinkedList<>();
		final Set<Object> processed = new HashSet<>();
		stack.addFirst(parentVertex);

		while (!stack.isEmpty()) {

			// plot(this.graph);

			final Object vertex = stack.peekFirst();

			final Object[] children = this.graph.getChildVertices(vertex);
			if (children.length > 0 && !processed.contains(vertex)) {
				for (final Object child : children) {
					stack.addFirst(child);
				}
				processed.add(vertex);
			} else {

				this.graph.getModel().beginUpdate();
				try {
					this.layout.setAverageNodeArea(1000000.0);
					this.layout.execute(vertex);
				} finally {
					this.graph.getModel().endUpdate();
				}

				this.graph.getModel().beginUpdate();
				try {
					resizeVertex(vertex, children);
				} finally {
					this.graph.getModel().endUpdate();
				}

				stack.removeFirst();
			}
		}
	}

	// @Override
	public void executeRec(final Object parentVertex) {

		plot(this.graph);

		final Object[] children = this.graph.getChildVertices(parentVertex);
		for (final Object child : children) {
			final mxRectangle before = this.graph.getCellBounds(child, false, false, true);
			execute(child);
			final mxRectangle after = this.graph.getCellBounds(child, false, false, true);
		}
		this.graph.getModel().beginUpdate();
		try {
			this.layout.execute(parentVertex);
		} finally {
			this.graph.getModel().endUpdate();
		}

		this.graph.getModel().beginUpdate();
		try {
			resizeVertex(parentVertex, children);
		} finally {
			this.graph.getModel().endUpdate();
		}
	}

	private static int counter = 0;

	private static void plot(final mxGraph g) {

		Document svgDoc = null;
		try {
			final DocumentBuilderFactory f = DocumentBuilderFactory.newInstance();
			final DocumentBuilder builder = f.newDocumentBuilder();
			svgDoc = builder.newDocument();
			final Element rootElement = svgDoc.createElement("svg");
			svgDoc.appendChild(rootElement);
		} catch (final ParserConfigurationException e) {
		}

		final mxSvgCanvas canvas = new mxSvgCanvas(svgDoc);
		g.drawGraph(canvas);
		final String svg = mxXmlUtils.getXml(canvas.getDocument());
		try (BufferedWriter bw = Files.newBufferedWriter(Paths.get("graphical_res_" + counter++ + ".svg"),
				StandardCharsets.UTF_8)) {
			bw.write(svg);
		} catch (final IOException e) {
			e.printStackTrace();
		}
	}

	private double getAvgVertexArea(final Object[] vertices) {

		if (vertices.length == 0) {
			return 160000.0;
		} else {
			double sumArea = 0.0;
			mxRectangle bounds;
			for (final Object v : vertices) {
				bounds = this.graph.getCellBounds(v);
				sumArea += bounds.getHeight() * bounds.getWidth();
			}
			return sumArea / vertices.length;
		}
	}

	private void resizeVertex(final Object vertex, final Object[] children) {

		mxRectangle newBounds = null;
		if (children.length > 0) {
			mxRectangle bounds;
			double minX = Double.POSITIVE_INFINITY;
			double maxX = Double.NEGATIVE_INFINITY;
			double minY = Double.POSITIVE_INFINITY;
			double maxY = Double.NEGATIVE_INFINITY;
			for (final Object child : children) {
				bounds = this.graph.getCellBounds(child);
				minX = Math.min(minX, bounds.getX());
				maxX = Math.max(maxX, bounds.getX() + bounds.getWidth());
				minY = Math.min(minY, bounds.getY());
				maxY = Math.max(maxY, bounds.getY() + bounds.getHeight());
			}
			double shiftX;
			double shiftY;
			bounds = this.graph.getCellBounds(vertex);
			if (bounds == null) {
				shiftX = minX - BORDER;
				shiftY = minY - BORDER;
			} else {
				shiftX = minX - bounds.getX() - BORDER;
				shiftY = minY - bounds.getY() - BORDER;
			}
			for (final Object child : children) {
				bounds = this.graph.getCellBounds(child);
				bounds.setX(bounds.getX() - shiftX);
				bounds.setY(bounds.getY() - shiftY);
				this.graph.resizeCell(child, bounds);
			}
			newBounds = new mxRectangle(minX, minY, maxX - minX + (2 * BORDER), maxY - minY + (2 * BORDER));
		} else {
			newBounds = this.graph.getBoundingBox(vertex);
		}

		this.graph.resizeCell(vertex, newBounds);
	}

	@Override
	public void moveCell(final Object cell, final double x, final double y) {
		this.layout.moveCell(cell, x, y);
	}

	@Override
	public Object getConstraint(final Object key, final Object cell) {
		return this.layout.getConstraint(key, cell);
	}

	@Override
	public Object getConstraint(final Object key, final Object cell, final Object edge, final boolean source) {
		return this.layout.getConstraint(key, cell, edge, source);
	}

	@Override
	public boolean isUseBoundingBox() {
		return this.layout.isUseBoundingBox();
	}

	@Override
	public void setUseBoundingBox(final boolean useBoundingBox) {
		this.layout.setUseBoundingBox(useBoundingBox);
	}

	@Override
	public boolean isVertexMovable(final Object vertex) {
		return this.layout.isVertexMovable(vertex);
	}

	@Override
	public boolean isVertexIgnored(final Object vertex) {
		return this.layout.isVertexIgnored(vertex);
	}

	@Override
	public boolean isEdgeIgnored(final Object edge) {
		return this.layout.isEdgeIgnored(edge);
	}

	@Override
	public void setEdgeStyleEnabled(final Object edge, final boolean value) {
		this.layout.setEdgeStyleEnabled(edge, value);
	}

	@Override
	public void setOrthogonalEdge(final Object edge, final boolean value) {
		this.layout.setOrthogonalEdge(edge, value);
	}

	@Override
	public mxPoint getParentOffset(final Object parentVertex) {
		return this.layout.getParentOffset(parentVertex);
	}

	@Override
	public void setEdgePoints(final Object edge, final List<mxPoint> points) {
		this.layout.setEdgePoints(edge, points);
	}

	@Override
	public mxRectangle getVertexBounds(final Object vertex) {
		return this.layout.getVertexBounds(vertex);
	}

	@Override
	public mxRectangle setVertexLocation(final Object vertex, final double x, final double y) {
		return this.layout.setVertexLocation(vertex, x, y);
	}

	@Override
	public void arrangeGroups(final Object[] groups, final int border) {
		this.layout.arrangeGroups(groups, border);
	}

}
