/**
 *
 */
package de.upb.pga3.panda2.extension.lvl2a;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import de.upb.pga3.panda2.core.datastructures.DetailLevel;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.FlowPath;
import de.upb.pga3.panda2.extension.lvl2a.flowpath.ResourceElement;
import de.upb.pga3.panda2.utilities.Constants;
import de.upb.pga3.panda2.utilities.HTMLFrameBuilder;
import soot.toolkits.scalar.Pair;

/**
 *
 * @author nptsy
 * @author Fabian
 * @author Monika
 */
public class ComparisonAnalysisResultLvl2a extends AnalysisResultLvl2a {

	// map of available paths from source
	// MultiMap<Permission, FlowPath> mMapSources;
	Map<ResourceElement, List<FlowPath>> mMapNewSources;

	Map<ResourceElement, List<FlowPath>> mMapRemovedSources;
	// map of available paths to sink
	// MultiMap<Permission, FlowPath> mMapSinks;
	Map<ResourceElement, List<FlowPath>> mMapNewSinks;

	Map<ResourceElement, List<FlowPath>> mMapRemovedSinks;

	// all filters for the user
	Map<String, ResourceElement> mMapComparisonFilters;

	public ComparisonAnalysisResultLvl2a(final AnalysisResultLvl2a inRes) {
		super();

		// Copy result
		this.mLstMessages = inRes.getMessages();
		this.mMapFilters = inRes.mMapFilters;
		this.mMapSinks = inRes.mMapSinks;
		this.mMapSources = inRes.mMapSources;

		// App name and app version
		setAppName(inRes.getAppName());

		// Initiate additional data structures
		this.mMapComparisonFilters = new HashMap<>();
		this.mMapNewSources = new HashMap<>();
		this.mMapRemovedSources = new HashMap<>();
		this.mMapNewSinks = new HashMap<>();
		this.mMapRemovedSinks = new HashMap<>();
	}

	/**
	 * Add new execution path
	 *
	 * @param inNewPath
	 *            new list of executed paths
	 *
	 */
	public boolean addNewPath(final FlowPath inNewPath) {
		if (inNewPath != null && inNewPath.isComplete()) {
			// add to map sources
			List<FlowPath> lstFlowPaths = this.mMapNewSources.get(inNewPath.getSource());
			if (lstFlowPaths == null) {
				lstFlowPaths = new ArrayList<>();
				lstFlowPaths.add(inNewPath);
				this.mMapNewSources.put(inNewPath.getSource(), lstFlowPaths);
			} else {
				lstFlowPaths.add(inNewPath);
			}

			// add to map sinks
			List<FlowPath> lstFlowPaths2 = this.mMapNewSinks.get(inNewPath.getSink());
			if (lstFlowPaths2 == null) {
				lstFlowPaths2 = new ArrayList<>();
				lstFlowPaths2.add(inNewPath);
				this.mMapNewSinks.put(inNewPath.getSink(), lstFlowPaths2);
			} else {
				lstFlowPaths2.add(inNewPath);
			}

			return true;
		} else {
			throw new IllegalArgumentException("The path to be added must not be null and must be complete!");
		}
	}

	/**
	 * add a path to the list of removed path
	 *
	 * @param inRemovedPath
	 *            the new removed path
	 * @return true if the new removed path is added successfully to the map,
	 *         otherwise false
	 */
	public boolean addRemovedPath(final FlowPath inRemovedPath) {
		if (inRemovedPath != null && inRemovedPath.isComplete()) {
			// add to map sources
			List<FlowPath> lstFlowPaths = this.mMapRemovedSources.get(inRemovedPath.getSource());
			if (lstFlowPaths == null) {
				lstFlowPaths = new ArrayList<>();
				lstFlowPaths.add(inRemovedPath);
				this.mMapRemovedSources.put(inRemovedPath.getSource(), lstFlowPaths);
			} else {
				lstFlowPaths.add(inRemovedPath);
			}

			// add to map sinks
			List<FlowPath> lstFlowPaths2 = this.mMapRemovedSinks.get(inRemovedPath.getSink());
			if (lstFlowPaths2 == null) {
				lstFlowPaths2 = new ArrayList<>();
				lstFlowPaths2.add(inRemovedPath);
				this.mMapRemovedSinks.put(inRemovedPath.getSink(), lstFlowPaths2);
			} else {
				lstFlowPaths2.add(inRemovedPath);
			}

			return true;
		} else {
			throw new IllegalArgumentException("The path to be added must not be null and must be complete!");
		}
	}

	@Override
	public void createFilters() {

		super.createFilters();

		fetchFilters(this.mMapNewSources.keySet(), Constants.PREFIX_SOURCE);
		fetchFilters(this.mMapRemovedSources.keySet(), Constants.PREFIX_SOURCE);
		fetchFilters(this.mMapNewSinks.keySet(), Constants.PREFIX_SINK);
		fetchFilters(this.mMapRemovedSinks.keySet(), Constants.PREFIX_SINK);
	}

	/**
	 * add permission to the map of filters
	 *
	 * @param permission
	 *            the set of permissions
	 * @param prefix
	 *            the prefix for permissions to make them as filters
	 */
	private void fetchFilters(final Collection<ResourceElement> permissions, final String prefix) {

		for (final ResourceElement p : permissions) {
			final String str = prefix + p.toString();
			if (!this.mMapComparisonFilters.containsKey(str)) {
				this.mMapComparisonFilters.put(str, p);
			}
		}
	}

	@Override
	public String getGraphicalResult(final DetailLevel inDetailLvl, final List<String> inFilters,
			final boolean inShowStats) {

		final Pair<Collection<FlowPath>, Collection<FlowPath>> pathsForRep = preprocess(inDetailLvl, inFilters);
		return createGraphicalResult(pathsForRep.getO1(), pathsForRep.getO2(), (DetailLevelLvl2a) inDetailLvl,
				inShowStats);
	}

	private Pair<Collection<FlowPath>, Collection<FlowPath>> preprocess(final DetailLevel inDetailLvl,
			final List<String> inFilters) {

		// validate the detail level
		DetailLevelLvl2a dlvl;
		if (inDetailLvl == null || !(inDetailLvl instanceof DetailLevelLvl2a)) {
			dlvl = DetailLevelLvl2a.RES_TO_RES;
		} else {
			dlvl = (DetailLevelLvl2a) inDetailLvl;
		}
		// process for creating graphical result
		// get paths to plot
		RepresentationPreprocessor rp = new RepresentationPreprocessor(this.mMapNewSources, this.mMapNewSinks,
				this.mMapComparisonFilters, inFilters, dlvl);
		final Collection<FlowPath> colNewFlowPaths = rp.preprocess();
		rp = new RepresentationPreprocessor(this.mMapRemovedSources, this.mMapRemovedSinks, this.mMapComparisonFilters,
				inFilters, dlvl);
		final Collection<FlowPath> colRemovedFlowPaths = rp.preprocess();

		return new Pair<>(colNewFlowPaths, colRemovedFlowPaths);
	}

	/**
	 * create graphical result of analysis
	 *
	 * @param inPaths
	 *            list of paths
	 * @return a string value of graphical result
	 */
	private String createGraphicalResult(final Collection<FlowPath> inNewPaths,
			final Collection<FlowPath> inRemovedPaths, final DetailLevelLvl2a inDetailLvl,
			final boolean autoHideHeader) {

		final StringBuilder sb = new StringBuilder();
		final HTMLFrameBuilder fb = new HTMLFrameBuilder(sb, "Intra-App Information Flow Analysis");

		fb.setAppTrustworthy(this.mMapSources.size() == 0);
		fb.addStaticticsRow("Number of new source permission(s): ",
				"<strong>" + this.mMapNewSources.size() + "</strong>");
		fb.addStaticticsRow("Number of removed source permission(s): ",
				"<strong>" + this.mMapRemovedSources.size() + "</strong>");
		fb.addStaticticsRow("Number of new sink permission(s): ", "<strong>" + this.mMapNewSinks.size() + "</strong>");
		fb.addStaticticsRow("Number of removed sink permission(s): ",
				"<strong>" + this.mMapRemovedSinks.size() + "</strong>");
		fb.addStaticticsRow("Number of new distinguish path(s): ", "<strong>" + inNewPaths.size() + "</strong>");
		fb.addStaticticsRow("Number of removed distinguish path(s): ",
				"<strong>" + inRemovedPaths.size() + "</strong>");

		AnalysisResultLvl2a.createLegend(inDetailLvl, fb);
		fb.addLegendEntry("<div style=\"background:" + SVGGraphBuilder.NEW_PATH_COLOR + ";\"></div>",
				"New Information Flow Edge");
		fb.addLegendEntry("<div style=\"background:" + SVGGraphBuilder.REMOVED_PATH_COLOR + ";\"></div>",
				"Removed Information Flow Edge");

		fb.setHeaderAutoHide(autoHideHeader);
		fb.setHintPersistent(false);
		fb.setCustomStyle(SVGGraphBuilder.GRAPH_STYLE);
		fb.setCustomScript(SVGGraphBuilder.GRAPH_SCRIPT);

		final SVGGraphBuilder builder = new SVGGraphBuilder();

		for (final FlowPath path : inNewPaths) {
			builder.addPath(path, SVGGraphBuilder.NEW_PATH_COLOR);
		}
		for (final FlowPath path : inRemovedPaths) {
			builder.addPath(path, SVGGraphBuilder.REMOVED_PATH_COLOR);
		}

		try {
			fb.append(SVGGraphBuilder.SVG_FILTER);
			fb.append(builder.buildGraph());
			fb.complete();
		} catch (final IOException e) {
			LOGGER.error("Error when creating graphical result: {}", e.getMessage());
			LOGGER.debug(e);
		}

		return sb.toString();
	}

	@Override
	public String getTextualResult(final DetailLevel inDetailLvl, final List<String> inFilters,
			final boolean inShowStats) {

		final Pair<Collection<FlowPath>, Collection<FlowPath>> pathsForRep = preprocess(inDetailLvl, inFilters);
		return createTextualResult(pathsForRep.getO1(), pathsForRep.getO2(), (DetailLevelLvl2a) inDetailLvl,
				inShowStats);
	}

	/**
	 * create textual string result
	 *
	 * @param inDetailLvl
	 * @param inFilters
	 * @return
	 */
	private String createTextualResult(final Collection<FlowPath> inNewPaths, final Collection<FlowPath> inRemovedPaths,
			final DetailLevelLvl2a inDetailLvl, final boolean autoHideHeader) {
		// add information of analysis: analysis mode and detail level
		final List<FlowPath> removedPaths = new LinkedList<>();
		for (final List<FlowPath> fpList : this.mMapRemovedSources.values()) {
			removedPaths.addAll(fpList);
		}
		final List<FlowPath> newPaths = new LinkedList<>();
		for (final List<FlowPath> fpList : this.mMapNewSources.values()) {
			newPaths.addAll(fpList);
		}

		/*
		 * process for statistic information of number sources and sinks of each
		 * app
		 */
		final int iNoNewSources = this.mMapNewSources.keySet().size();
		final int iNoNewSinks = this.mMapNewSinks.keySet().size();
		final int iNoRemovedSources = this.mMapRemovedSources.keySet().size();
		final int iNoRemoveSinks = this.mMapRemovedSinks.keySet().size();

		final StringBuilder sb = new StringBuilder();
		final HTMLFrameBuilder fb = new HTMLFrameBuilder(sb, "Intra-App Information Flow Analysis");

		// statistic
		fb.addStaticticsRow("Number of source permission(s): ",
				"<font color=\"red\"><strong>" + iNoNewSources + "</strong></font>");
		fb.addStaticticsRow("Number of removed source permission(s): ",
				"<font color=\"red\"><strong>" + iNoRemovedSources + "</strong></font>");
		fb.addStaticticsRow("Number of sink permission(s): ",
				"<font color=\"blue\"><strong>" + iNoNewSinks + "</strong></font>");
		fb.addStaticticsRow("Number of removed sink permission(s): ",
				"<font color=\"blue\"><strong>" + iNoRemoveSinks + "</strong></font>");
		fb.addStaticticsRow("Number of new path(s): ", "<strong>" + newPaths.size() + "</strong>");
		fb.addStaticticsRow("Number of removed path(s): ", "<strong>" + removedPaths.size() + "</strong>");

		// legend
		fb.addLegendEntry("<div class=\"icon\" style=\"background:#ff0000;\"></div>", "SOURCE PERMISSION");
		fb.addLegendEntry("<div class=\"icon\" style=\"background:#0006fd;\"></div>", "SINK PERMISSION");

		fb.setHeaderAutoHide(autoHideHeader);
		fb.setHintPersistent(false);
		fb.setCustomStyle(HtmlTableBuilder.cssStyle);

		final HtmlTableBuilder tableBuilder = new HtmlTableBuilder(true, getAppName());

		tableBuilder.processResult(inNewPaths, inRemovedPaths, inDetailLvl);

		try {
			fb.append(tableBuilder.toString());
			fb.complete();
		} catch (final IOException e) {
			e.printStackTrace();
		}

		return sb.toString();

	}

	@Override
	public List<String> getFilters() {

		final List<String> filters = new ArrayList<>();
		filters.addAll(this.mMapComparisonFilters.keySet());
		return filters;

	}

}
