/**
 *
 */
package de.upb.pga3.panda2.extension.lvl2a;

import java.util.Set;

import de.upb.pga3.panda2.core.datastructures.AnalysisGraph;
import de.upb.pga3.panda2.core.datastructures.Input;
import de.upb.pga3.panda2.core.datastructures.Transition;
import gnu.trove.strategy.HashingStrategy;
import gnu.trove.strategy.IdentityHashingStrategy;
import soot.SootMethod;
import soot.Unit;
import soot.util.HashMultiMap;
import soot.util.MultiMap;

/**
 *
 * @author nptsy
 * @author Fabian
 */
public class AnalysisGraphLvl2a extends AnalysisGraph {

	private transient final MultiMap<Unit, ParameterNode> mUnitToParamNodes;
	private transient final MultiMap<SootMethod, ParameterNode> mMethodToParamNodes;

	/**
	 * Constructor
	 *
	 * @param inInput
	 *            input object
	 */
	public AnalysisGraphLvl2a(final Input inInput) {
		super(inInput, new IdentityHashingStrategy<>(), new EquivToHashingStrategy());
		this.mUnitToParamNodes = new HashMultiMap<>();
		this.mMethodToParamNodes = new HashMultiMap<>();
	}

	/**
	 * Adds a parameter node to the graph
	 *
	 * @param inParamNode
	 *            The node to be added
	 */
	public void addParameterNode(final ParameterNode inParamNode) {
		if (inParamNode.belongsToMethod()) {
			this.mMethodToParamNodes.put(inParamNode.getCorrespondingMethod(), inParamNode);
		} else {
			this.mUnitToParamNodes.put(inParamNode.getCorrespondingUnit(), inParamNode);
		}

	}

	/**
	 * Gives the set of statements that have associated parameter nodes
	 *
	 * @return The set of statements with parameter nodes
	 */
	public Set<Unit> getParameterizedInvokeStmnts() {
		return this.mUnitToParamNodes.keySet();
	}

	/**
	 * Gives the set of methods that have associated parameter nodes
	 *
	 * @return The set of methods with parameter nodes
	 */
	public Set<SootMethod> getParameterizedMethods() {
		return this.mMethodToParamNodes.keySet();
	}

	/**
	 * States whether parameter nodes correspond to the given statement
	 *
	 * @param inStmnt
	 *            The statement to be tested
	 * @return {@code true} if and only if the given statement has parameter
	 *         nodes
	 */
	public boolean hasParameterNodes(final Unit inStmnt) {
		return this.mUnitToParamNodes.containsKey(inStmnt);
	}

	/**
	 * States whether parameter nodes correspond to the given method
	 *
	 * @param inMethod
	 *            The method to be tested
	 * @return {@code true} if and only if the given method has parameter nodes
	 */
	public boolean hasParameterNodes(final SootMethod inMethod) {
		return this.mMethodToParamNodes.containsKey(inMethod);
	}

	/**
	 * Gives all parameter nodes that correspond to the given statement in which
	 * a method is invoked
	 *
	 * @param inInvokeStmnt
	 *            The statement to get the parameter nodes for
	 * @return The set of parameter nodes for the given statement
	 */
	public Set<ParameterNode> getParameterNodes(final Unit inInvokeStmnt) {
		if (this.mUnitToParamNodes.containsKey(inInvokeStmnt)) {
			return this.mUnitToParamNodes.get(inInvokeStmnt);
		} else {
			throw new IllegalArgumentException("No parameter nodes correspond to the given statement!");
		}
	}

	/**
	 * Gives all parameter nodes that correspond to the given method
	 *
	 * @param inMethod
	 *            The method to get the parameter nodes for
	 * @return The set of parameter nodes for the given method
	 */
	public Set<ParameterNode> getParameterNodes(final SootMethod inMethod) {
		if (this.mMethodToParamNodes.containsKey(inMethod)) {
			return this.mMethodToParamNodes.get(inMethod);
		} else {
			throw new IllegalArgumentException("No parameter nodes correspond to the given statement!");
		}
	}

	@SuppressWarnings("serial")
	private static class EquivToHashingStrategy implements HashingStrategy<Transition> {

		public EquivToHashingStrategy() {
			// Do nothing
		}

		@Override
		public int computeHashCode(final Transition arg0) {
			return ((TransitionLvl2a) arg0).equivHashCode();
		}

		@Override
		public boolean equals(final Transition arg0, final Transition arg1) {
			return ((TransitionLvl2a) arg0).equivTo(arg1);
		}

	}
}
