package de.upb.pga3.panda2.core.services;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author RamKumar
 */

public class XMLNode extends XMLElement {

	public String tag;

	XMLNode parent = null;

	ArrayList<XMLNode> children = null;

	/* Mapping of Attributes */
	Map<String, XMLAttribute<?>> attributes = null;

	public XMLNode(final String tag, final String ns, final XMLNode parent) {
		this(tag, ns, parent, true);
	}

	public XMLNode(final String tag, final String ns, final XMLNode parent, final boolean added) {
		super(ns, added);
		this.tag = tag;
		this.parent = parent;
		if (parent != null) {
			parent.addChild(this);
		}
	}

	/**
	 *
	 * @return Node's tag
	 */
	public String getTag() {
		return this.tag;
	}

	/**
	 *
	 * @param sibling
	 *            - sibling XML node
	 * @return true if sibling added, false otherwise
	 */
	public boolean addSiblingBefore(final XMLNode sibling) {
		if (this.parent != null) {
			this.parent.addChild(sibling, this.parent.getChildren().indexOf(this));
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Add Sibling XML node.
	 *
	 * @param sibling
	 *            - sibling XML node
	 * @return true if sibling added, false otherwise
	 */
	public boolean addSiblingAfter(final XMLNode sibling) {
		if (this.parent != null) {
			this.parent.addChild(sibling, this.parent.getChildren().indexOf(this) + 1);
			return true;
		} else {
			return false;
		}
	}

	/**
	 * add child XMLNode
	 *
	 * @param child
	 *            XML node
	 * @return XML node
	 */
	public XMLNode addChild(final XMLNode child) {
		if (this.children == null) {
			this.children = new ArrayList<XMLNode>();
		}
		this.children.add(child);
		return this;
	}

	/**
	 * Add child XMLNode
	 *
	 * @param child
	 *            XML node
	 * @param index
	 *            where node to be added
	 * @return XML node
	 */
	public XMLNode addChild(final XMLNode child, final int index) {
		if (this.children == null) {
			this.children = new ArrayList<XMLNode>();
		}
		this.children.add(index, child);
		return this;
	}

	/**
	 *
	 * @return List of child XMLNodes
	 */
	public List<XMLNode> getChildren() {
		if (this.children == null) {
			return Collections.emptyList();
		}
		return new ArrayList<XMLNode>(this.children);
	}

	/**
	 *
	 * @param tag
	 *            Node's tag
	 * @return List of child XMLNodes
	 */
	public List<XMLNode> getChildrenWithTag(final String tag) {
		if (this.children == null) {
			return Collections.emptyList();
		}

		final ArrayList<XMLNode> children = new ArrayList<XMLNode>();
		for (final XMLNode child : this.children) {
			if (child.getTag().equals(tag)) {
				children.add(child);
			}
		}

		return children;
	}

	/**
	 * Return attribute for a node
	 *
	 * @return XMLAttribute
	 */
	public Map<String, XMLAttribute<?>> getAttributes() {
		if (this.attributes == null) {
			return Collections.emptyMap();
		}
		return new HashMap<String, XMLAttribute<?>>(this.attributes);
	}

	/**
	 *
	 * @param name
	 *            Attribute name
	 * @return true if node has attribute, false otherwise
	 */
	public boolean hasAttribute(final String name) {
		if (this.attributes == null) {
			return false;
		}
		return this.attributes.containsKey(name);
	}

	/**
	 * Get attribute by name
	 *
	 * @param name
	 *            attribute name
	 * @return XMLAttribute
	 */
	public XMLAttribute<?> getAttribute(final String name) {
		if (this.attributes == null) {
			return null;
		}
		return this.attributes.get(name);
	}

	/**
	 * Add attribute to the node
	 *
	 * @param attr
	 *            XMLAttribute
	 */
	public void addAttribute(final XMLAttribute<?> attr) {
		if (attr == null) {
			throw new NullPointerException("XMLAttribute is null");
		}

		if (this.attributes == null) {
			this.attributes = new HashMap<String, XMLAttribute<?>>();
		}
		this.attributes.put(attr.getName(), attr);
	}

	/**
	 * Get Node's parent
	 *
	 * @return XMLNode parent node
	 */
	public XMLNode getParent() {
		return this.parent;
	}

	/**
	 * Set parent for XML node
	 *
	 * @param parent
	 *            XML node
	 */
	public void setParent(final XMLNode parent) {
		this.parent = parent;
	}

}
