/**
 *
 */
package de.upb.pga3.panda2.core.services;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import de.upb.pga3.panda2.client.core.datastructures.ManifestInfo;
import de.upb.pga3.panda2.core.datastructures.IntentFilter;
import de.upb.pga3.panda2.utilities.Constants;

/**
 * ManifestXMLParser class is a single instance that processes for manifest file
 *
 * @author nptsy
 */
public class ManifestXMLParser extends BinaryParser {

	// logger
	private static final Logger LOGGER = LogManager.getLogger(ManifestXMLParser.class);

	// flag for checking whether an apk is inputed.
	// private static boolean isAlreadyFetchingFile = false;

	// String content in xml file
	private String mStrXMLContent;

	// manifest node
	// private Node mNodeManifest;
	private ManifestInfo mManifestInfo;

	// application list node
	private String mStrAppName;

	// list of activities element
	private List<String> mLstActivities;

	// list of services
	private List<String> mLstServices;

	// list of receivers
	private List<String> mLstReceivers;

	// list of provider
	private List<String> mLstProviders;

	// list of uses permissions
	private List<String> mLstUsesPermission;

	// map of required permission. This is the permission defined inside element
	// <\activity> (<\receiver>, <\service>, <\provider>) with the attribute
	// [android:permission]
	private Map<String, String> mMapRequiredPermissions;

	private Map<String, String> mMapProviderURIs;

	// list of intent filters
	private List<IntentFilter> mLstIntentFilter;

	// list of services
	// list of broadcast receiver
	// list of content provider

	/**
	 * private constructor
	 */
	public ManifestXMLParser(final String inAPKPath) {
		// parse data
		super(inAPKPath);
	}

	/**
	 * read data in manifest xml file in the input .apk
	 *
	 * @param inAPKFile
	 *            the apk file path
	 * @return true if successful otherwise false
	 */
	@Override
	protected boolean parseData(final String inAPKFile) {

		final File apkFile = new File(inAPKFile);
		if (!apkFile.exists()) {
			LOGGER.info("Can not find the apk file: " + inAPKFile);
			return false;
		}

		// --------------------------------------------------------
		// **** This is reserved for initilization of variables
		this.mLstActivities = new ArrayList<>();
		this.mLstServices = new ArrayList<>();
		this.mLstProviders = new ArrayList<>();
		this.mLstReceivers = new ArrayList<>();
		this.mLstUsesPermission = new ArrayList<>();
		this.mMapRequiredPermissions = new HashMap<>();
		this.mLstIntentFilter = new ArrayList<>();
		this.mMapProviderURIs = new HashMap<>();
		// --------------------------------------------------------

		this.mStrXMLContent = getStrXMLFromAPK(apkFile, Constants.MANIFEST_FILE);

		// processing data in manifest file
		// crete document builder factory
		final DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();

		// create document builder
		DocumentBuilder builder;
		try {
			builder = builderFactory.newDocumentBuilder();

			// parse document object from xml string
			final Document doc = builder
					.parse(new InputSource(new ByteArrayInputStream(this.mStrXMLContent.getBytes("utf-8"))));

			// ****** get all element and attribute here ******\\
			// -- [start] get <manifest> element
			final NodeList manifests = doc.getElementsByTagName(Constants.TAG_MANIFEST);
			if (manifests.getLength() < 1) {
				// throw RuntimeException
				throw new RuntimeException("Invalid manifest file. Manifest file contains no <manifest> element");
			} else {
				if (manifests.getLength() > 1) {
					// throw RuntimeException
					throw new RuntimeException(
							"Invalid manifest file.	Manifest file contains more than one <manifest> element");
				}
			}
			extractManifestInfo(manifests.item(0));
			// -- [end]

			// -- [start] get <application> node ==> for android app
			// name and enabled state
			final NodeList applications = doc.getElementsByTagName(Constants.TAG_APPLICATION);
			if (applications.getLength() < 1) {
				throw new RuntimeException("Invalid manifest file. Manifest file contains no <application> element");
			} else {
				if (applications.getLength() > 1) {
					throw new RuntimeException(
							"Invalid manifest file. Manifest file contains more than one <application> element");
				}
			}

			// extractAppName(applications.item(0));
			extractAppName();
			// -- [end]

			// -- [start]get <uses-permission> nodes
			final NodeList usesPermission = doc.getElementsByTagName(Constants.TAG_USES_PERMISSION);
			extractUsesPermission(usesPermission);
			// -- [end]

			// -- [start] get <activity> nodes
			final NodeList lstActivities = doc.getElementsByTagName(Constants.TAG_ACTIVITY);
			extractActivityInfo(lstActivities);
			// -- [end]

			// -- [start] get <service> nodes
			final NodeList lstServices = doc.getElementsByTagName(Constants.TAG_SERVICE);
			extractServiceInfo(lstServices);
			// -- [end]

			// -- [start] get <provider> nodes
			final NodeList lstProvider = doc.getElementsByTagName(Constants.TAG_PROVIDER);
			extractProviderInfo(lstProvider);
			// -- [end]

			// -- [start] get <receiver> nodes
			final NodeList lstReceiver = doc.getElementsByTagName(Constants.TAG_RECEIVER);
			extractReceiverInfo(lstReceiver);
			// -- [end]

			// -- [start] get any other elements
			// -- [end]

		} catch (final RuntimeException e) {
			LOGGER.info("Exception: " + e.getMessage());
			e.printStackTrace();
		} catch (final ParserConfigurationException e) {
			LOGGER.info("Exception: " + e.getMessage());
			e.printStackTrace();
		} catch (final UnsupportedEncodingException e) {
			LOGGER.info("Exception: " + e.getMessage());
			e.printStackTrace();
		} catch (final SAXException e) {
			LOGGER.info("Exception: " + e.getMessage());
			e.printStackTrace();
		} catch (final IOException e) {
			LOGGER.info("Exception: " + e.getMessage());
			e.printStackTrace();
		}

		return true;
	}

	/**
	 * extract activity info defined in element <\activity> with the attribute
	 * [android:name]. The info includes: list of activities, required
	 * permission for each activity and the intent filter of each activity
	 *
	 * @param inLstActivities
	 *            list of element <\activity>
	 */
	private void extractActivityInfo(final NodeList inLstActivities) {

		try {
			if (inLstActivities != null) {
				final int iSize = inLstActivities.getLength();
				for (int i = 0; i < iSize; i++) {
					final Node activity = inLstActivities.item(i);

					final String strActivity = activity.getAttributes().getNamedItem(Constants.ATT_ANDROID_NAME)
							.getNodeValue();
					// list of activity
					this.mLstActivities.add(strActivity);

					final Node reqPermission = activity.getAttributes().getNamedItem(Constants.ATT_ANDROID_PERMISSION);
					if (reqPermission != null) {
						// get required permission here
						final String requiredPermission = reqPermission.getNodeValue();

						// map of required permissions for activity
						this.mMapRequiredPermissions.put(strActivity, requiredPermission);
					}

					// we can get intent filter here
					final NodeList childLst = activity.getChildNodes();
					int iSizeChildLst = 0;
					if (childLst != null && (iSizeChildLst = childLst.getLength()) > 0) {
						for (int j = 0; j < iSizeChildLst; j++) {
							final Node childNode = childLst.item(j);
							if (childNode.getNodeName().equals(Constants.TAG_INTENT_FILTER)) {
								extractIntentFilter(strActivity, childNode);
							}
						}
					}
				}
			}
		} catch (final Exception ex) {
			LOGGER.info("Exception: " + ex.getMessage());
			ex.printStackTrace();
		}
	}

	/**
	 * extract app name in element <\application> with attribute [android:label]
	 *
	 * @param inNodeApp
	 *            the element <\application>
	 */
	private void extractAppName() {
		final String pkgName = this.mManifestInfo.getPackage();
		final int pos = pkgName.lastIndexOf(".");
		this.mStrAppName = pkgName.substring(pos + 1, pkgName.length());
	}

	/**
	 * extract information of intent filter for activity, activity-alias,
	 * service or receiver
	 *
	 * @param inEleName
	 * @param inNodeIntentFilter
	 */
	private void extractIntentFilter(final String inEleName, final Node inNodeIntentFilter) {

		try {
			final IntentFilter intent = new IntentFilter(inEleName);
			// list of childrend that contains action, category and data
			final NodeList childLst = inNodeIntentFilter.getChildNodes();
			int iSize = 0;
			if (childLst != null && (iSize = childLst.getLength()) > 0) {
				for (int i = 0; i < iSize; i++) {
					final Node node = childLst.item(i);
					if (node.getNodeName().equals(Constants.TAG_ACTION)) {
						// tag action
						final String actionName = node.getAttributes().getNamedItem(Constants.ATT_ANDROID_NAME)
								.getNodeValue();

						intent.setActionName(actionName);
					} else {

						// process for element category
						if (node.getNodeName().equals(Constants.TAG_CATEGORY)) {
							final String categoryName = node.getAttributes().getNamedItem(Constants.ATT_ANDROID_NAME)
									.getNodeValue();
							intent.setCategoryName(categoryName);
						} else {

							// process for element data
							if (node.getNodeName().equals(Constants.TAG_DATA)) {
								final int iLength = node.getAttributes().getLength();
								if (iLength > 0) {
									for (int j = 0; j < iLength; j++) {
										final Node attData = node.getAttributes().item(j);
										final String attrName = attData.getNodeName();
										final String attrValue = attData.getNodeValue();

										intent.addData(attrName, attrValue);
									}
								}
							}
						}
					}
				}
				this.mLstIntentFilter.add(intent);
			}
		} catch (final Exception ex) {
			LOGGER.info("Exception: " + ex.getMessage());
			ex.printStackTrace();
		}
	}

	/**
	 * get manifest information
	 *
	 * @param inManifestNode
	 */
	private void extractManifestInfo(final Node inManifestNode) {
		if (inManifestNode != null) {
			try {
				// get package name defined in element <\manifest> with the
				// attribute [package]
				final String packName = inManifestNode.getAttributes().getNamedItem(Constants.ATT_PACKAGE)
						.getNodeValue();

				// get version code defined in element <\manifest> with the
				// attribute [android:versionCode]
				final String versionCode = inManifestNode.getAttributes()
						.getNamedItem(Constants.ATT_ANDROID_VERSION_CODE).getNodeValue();

				// get version name defined in element <\manifest> with the
				// attribute [android:versionName]
				final String versionName = inManifestNode.getAttributes()
						.getNamedItem(Constants.ATT_ANDROID_VERSION_NAME).getNodeValue();

				this.mManifestInfo = new ManifestInfo(versionCode, versionName, packName);
			} catch (final Exception ex) {
				LOGGER.info("Exception: " + ex.getMessage());
				ex.printStackTrace();
			}
		}
	}

	/**
	 * extract provider information. It includes the provider name, required
	 * permissions and the URI. The URI is got in
	 *
	 * @param inLstProviders
	 */
	private void extractProviderInfo(final NodeList inLstProviders) {
		try {
			if (inLstProviders != null) {
				final int iSize = inLstProviders.getLength();
				for (int i = 0; i < iSize; i++) {
					final Node provider = inLstProviders.item(i);

					final String strProvider = provider.getAttributes().getNamedItem(Constants.ATT_ANDROID_NAME)
							.getNodeValue();
					// list of providers
					this.mLstProviders.add(strProvider);

					// extract required permission
					final Node reqPermission = provider.getAttributes().getNamedItem(Constants.ATT_ANDROID_PERMISSION);
					if (reqPermission != null) {
						// get required permission here
						final String requiredPermission = reqPermission.getNodeValue();

						// map of required permissions for activity
						this.mMapRequiredPermissions.put(strProvider, requiredPermission);
					}

					// extract provider URIs
					final NodeList childLst = provider.getChildNodes();
					int iSizeChildLst = 0;
					if (childLst != null && (iSizeChildLst = childLst.getLength()) > 0) {
						for (int j = 0; j < iSizeChildLst; j++) {
							final Node child = childLst.item(j);
							if (child.getNodeName().equals(Constants.TAG_GRANT_URI_PERMISSION)) {
								final Node attrPath = child.getAttributes().getNamedItem(Constants.ATT_ANDROID_PATH);
								if (attrPath != null) {
									this.mMapProviderURIs.put(strProvider, attrPath.getNodeValue());
								}
							}
						}
					}
				}
			}
		} catch (final Exception ex) {
			LOGGER.info("Exception: " + ex.getMessage());
			ex.printStackTrace();
		}
	}

	private void extractReceiverInfo(final NodeList inLstReceivers) {
		try {
			if (inLstReceivers != null) {
				final int iSize = inLstReceivers.getLength();
				for (int i = 0; i < iSize; i++) {
					final Node activity = inLstReceivers.item(i);

					final String strReceiver = activity.getAttributes().getNamedItem(Constants.ATT_ANDROID_NAME)
							.getNodeValue();
					// list of receivers
					this.mLstReceivers.add(strReceiver);
					final Node reqPermission = activity.getAttributes().getNamedItem(Constants.ATT_ANDROID_PERMISSION);
					if (reqPermission != null) {
						// get required permission here
						final String requiredPermission = reqPermission.getNodeValue();

						// map of required permissions for activity
						this.mMapRequiredPermissions.put(strReceiver, requiredPermission);
					}

					// we can get intent filter here
					final NodeList childLst = activity.getChildNodes();
					int iSizeChildLst = 0;
					if (childLst != null && (iSizeChildLst = childLst.getLength()) > 0) {
						for (int j = 0; j < iSizeChildLst; j++) {
							final Node childNode = childLst.item(j);
							if (childNode.getNodeName().equals(Constants.TAG_INTENT_FILTER)) {
								extractIntentFilter(strReceiver, childNode);
							}
						}
					}
				}
			}
		} catch (final Exception ex) {
			LOGGER.info("Exception: " + ex.getMessage());
			ex.printStackTrace();
		}
	}

	private void extractServiceInfo(final NodeList inLstServices) {

		try {
			if (inLstServices != null) {
				final int iSize = inLstServices.getLength();
				for (int i = 0; i < iSize; i++) {
					final Node activity = inLstServices.item(i);

					final String strService = activity.getAttributes().getNamedItem(Constants.ATT_ANDROID_NAME)
							.getNodeValue();
					// list of services
					this.mLstServices.add(strService);
					// get required permission here
					final Node reqPermission = activity.getAttributes().getNamedItem(Constants.ATT_ANDROID_PERMISSION);
					if (reqPermission != null) {
						final String requiredPermission = reqPermission.getNodeValue();

						// map of required permissions for activity
						this.mMapRequiredPermissions.put(strService, requiredPermission);
					}
					// we can get intent filter here
					final NodeList childLst = activity.getChildNodes();
					int iSizeChildLst = 0;
					if (childLst != null && (iSizeChildLst = childLst.getLength()) > 0) {
						for (int j = 0; j < iSizeChildLst; j++) {
							final Node childNode = childLst.item(j);
							if (childNode.getNodeName().equals(Constants.TAG_INTENT_FILTER)) {
								extractIntentFilter(strService, childNode);
							}
						}
					}
				}
			}
		} catch (final Exception ex) {
			LOGGER.info("Exception: " + ex.getMessage());
			ex.printStackTrace();
		}
	}

	/**
	 * extract uses permission defined in element </uses-permission> with
	 * attribute [android:name]
	 *
	 * @param inNodeListPermission
	 *            list of elements </uses-permission>
	 */
	private void extractUsesPermission(final NodeList inNodeListPermission) {
		try {
			if (inNodeListPermission != null) {
				final int iSize = inNodeListPermission.getLength();
				for (int i = 0; i < iSize; i++) {
					final Node nPerm = inNodeListPermission.item(i);
					final String strPerm = nPerm.getAttributes().getNamedItem(Constants.ATT_ANDROID_NAME)
							.getNodeValue();
					this.mLstUsesPermission.add(strPerm);
				}
			}
		} catch (final Exception ex) {
			LOGGER.info("Exception: " + ex.getMessage());
			ex.printStackTrace();
		}
	}

	/**
	 * get list of providers' name defined in element <\provider> with the
	 * attribute [android:name]. This name contains also the package name
	 *
	 * @return
	 */
	public List<String> getLstProviders() {
		// valid mLstProviders
		if (this.mLstProviders != null && this.mLstProviders.size() > 0) {

			/*
			 * full array list in case of the original list does not contain
			 * package name
			 */
			final List<String> fullNameProviders = new ArrayList<>();

			// package name got from manifest info object
			final String packageName = this.mManifestInfo.getPackage();
			int iCount = 0;
			for (final String proName : this.mLstProviders) {
				if (proName.contains(packageName)) {
					break;
				} else {
					fullNameProviders.add(packageName + proName);
					iCount++;
				}
			}

			if (iCount == 0) {
				return this.mLstProviders;
			} else {
				return fullNameProviders;
			}
		}
		return null;
	}

	/**
	 * get list of receivers' name defined in element <\receivers> with the
	 * attribute [android:name]. This name contains also the package name
	 *
	 * @return
	 */
	public List<String> getLstReceivers() {
		// valid mLstReceivers
		if (this.mLstReceivers != null && this.mLstReceivers.size() > 0) {

			/*
			 * full array list in case of the original list does not contain
			 * package name
			 */
			final List<String> fullNameReceivers = new ArrayList<>();

			// package name got from manifest info object
			final String packageName = this.mManifestInfo.getPackage();
			int iCount = 0;
			for (final String recName : this.mLstReceivers) {
				if (recName.contains(packageName)) {
					break;
				} else {
					fullNameReceivers.add(packageName + recName);
					iCount++;
				}
			}

			if (iCount == 0) {
				return this.mLstReceivers;
			} else {
				return fullNameReceivers;
			}
		}
		return null;
	}

	public List<String> getLstServices() {
		// valid mLstServices
		if (this.mLstServices != null && this.mLstServices.size() > 0) {

			/*
			 * full array list in case of the original list does not contain
			 * package name
			 */
			final List<String> fullNameServices = new ArrayList<>();

			// package name got from manifest info object
			final String packageName = this.mManifestInfo.getPackage();
			int iCount = 0;
			for (final String serName : this.mLstServices) {
				if (serName.contains(packageName)) {
					break;
				} else {
					fullNameServices.add(packageName + serName);
					iCount++;
				}
			}

			if (iCount == 0) {
				return this.mLstServices;
			} else {
				return fullNameServices;
			}
		}
		return null;
	}

	/**
	 * get list of activities' name defined in element <\activity> with the
	 * attribute [android:name]. This name contains also the package name
	 *
	 * @return list of activities
	 */
	public List<String> getLstActivities() {
		// valid mLstActivities
		if (this.mLstActivities != null && this.mLstActivities.size() > 0) {

			/*
			 * full array list in case of the original list does not contain
			 * package name
			 */
			final List<String> fullNameActivities = new ArrayList<>();

			// package name got from manifest info object
			final String packageName = this.mManifestInfo.getPackage();
			int iCount = 0;
			for (final String actName : this.mLstActivities) {
				if (actName.contains(packageName)) {
					break;
				} else {
					fullNameActivities.add(packageName + actName);
					iCount++;
				}
			}

			if (iCount == 0) {
				return this.mLstActivities;
			} else {
				return fullNameActivities;
			}
		}
		return null;
	}

	/**
	 * get manifest information which is defined in element <\manifest>
	 *
	 * @return object ManifestInfo that contains information of app
	 */
	public ManifestInfo getManifestInfo() {
		return this.mManifestInfo;
	}

	/**
	 * get app name from the file name
	 *
	 * @return
	 */
	public String getAppName() {
		return this.mStrAppName;
	}

	/**
	 * get list of uses permission
	 *
	 * @return a list of uses permission
	 */
	public List<String> getUsesPermission() {
		return this.mLstUsesPermission;
	}

	/**
	 * get required permission for each components defined in manifest file. In
	 * the map, key is the name of android component and value is the permission
	 * used by the component
	 *
	 * @return map of required permissions
	 */

	public Map<String, String> getRequiredPermissions() {
		return this.mMapRequiredPermissions;
	}

	/**
	 * get intent-filter of element <\activity>, <\ativity-alias>,
	 * <\service> and <\receiver>
	 *
	 * @return map of intent filter
	 */
	public List<IntentFilter> getIntentFilters() {
		return this.mLstIntentFilter;
	}

	/**
	 * get content provider URIs which is defined in element
	 * <\grant-uri-permission> in attribute [android:path]. Until now, provider
	 * just consists of only one element <\grant-uri-permisison>, we dont need a
	 * List of string URIs
	 *
	 * @return map of provider URIs
	 */
	public Map<String, String> getContentProviderURIs() {
		return this.mMapProviderURIs;
	}
}
