package de.upb.pga3.panda2.core.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import soot.SootClass;
import soot.Unit;

/**
 * All information gathered by the {@link StatementAnalyzer} will be stored in
 * objects of this class. It provides information for any type of intent and
 * includes if possible a list of targets of the associated intent.
 *
 * @author Fabian
 * @author Felix
 *
 */
public class IntentInformation {
	/**
	 * Constants that determine the type of intent
	 */
	public static final int TYPE_UNKNOWN = -1;
	public static final int TYPE_EXPLICIT = 0;
	public static final int TYPE_IMPLICIT = 1;

	private String classNameOrActionString;
	private Collection<SootClass> mTargetClasses;
	private List<Unit> mLstLaunchingUnits;

	private int mTypeIntent = -1;
	private boolean mIsValid = false;

	/**
	 * Constructor
	 *
	 * @param inType
	 */
	public IntentInformation(final int inType) {
		this.classNameOrActionString = "";
		this.mTypeIntent = inType;
		this.mTargetClasses = null;
		this.mLstLaunchingUnits = new ArrayList<>();
		this.mTargetClasses = null;
	}

	/**
	 * add the target classes in type of SootClass
	 *
	 * @param classes
	 */
	public void addTargetClasses(final Collection<SootClass> classes) {
		if (this.mTargetClasses == null) {
			this.mTargetClasses = classes;
		} else {
			this.mTargetClasses.addAll(classes);
		}
	}

	/**
	 * get list of target classes
	 *
	 * @return a collection of target classes in type of SootClass
	 */
	public Collection<SootClass> getTargetClasses() {
		return this.mTargetClasses;
	}

	/**
	 * get launching units (statements)
	 *
	 * @return a list of launching units
	 */
	public List<Unit> getLstLaunchingUnits() {
		if (this.mLstLaunchingUnits != null && !this.mLstLaunchingUnits.isEmpty()) {
			return this.mLstLaunchingUnits;
		}

		return null;
	}

	/**
	 * set type of intent (explicit, implicit or unknown)
	 *
	 * @param inType
	 *            type of intent
	 */
	public void setTypeIntent(final int inType) {
		this.mTypeIntent = inType;
	}

	/**
	 * get type of intent
	 *
	 * @return type of intent (-1: unknown, 0: explicit and 1: implicit)
	 */
	public int getTypeIntent() {
		return this.mTypeIntent;
	}

	/**
	 * set valid if an intent is started
	 *
	 * @param inIsValid
	 *            true if being started otherwise false
	 */
	public void setValid(final boolean inIsValid) {
		this.mIsValid = inIsValid;
	}

	/**
	 * check if an intent is valid
	 *
	 * @return true if yes otherwise false
	 */
	public boolean isValid() {
		return this.mIsValid;
	}

	/**
	 * add starting units for intents
	 *
	 * @param inUnit
	 */
	public void addStartingUnit(final Unit inUnit) {
		if (this.mLstLaunchingUnits == null) {
			this.mLstLaunchingUnits = new ArrayList<>();
		}
		this.mLstLaunchingUnits.add(inUnit);
	}

	/**
	 * get target class name or action string
	 *
	 * @return
	 */
	public String getClassNameOrActionString() {
		return this.classNameOrActionString;
	}

	/**
	 * set class name or action string
	 *
	 * @param classNameOrActionString
	 */
	public void setClassNameOrActionString(final String classNameOrActionString) {
		this.classNameOrActionString = classNameOrActionString;
	}

	@Override
	public String toString() {
		String intentTypeStr;
		String classNameOrActionStr;
		switch (this.mTypeIntent) {
		case 0:
			intentTypeStr = "Explicit";
			classNameOrActionStr = "Classname";
			break;
		case 1:
			intentTypeStr = "Implicit";
			classNameOrActionStr = "Action String";
			break;
		default:
			intentTypeStr = "Unknown";
			classNameOrActionStr = "Classname or Action String";
			break;
		}

		String output = "*** " + intentTypeStr + " Intent ***\n" + classNameOrActionStr + ": "
				+ this.classNameOrActionString + "\nStarting unit: " + this.mLstLaunchingUnits + "\nList of targets:\n";
		for (final SootClass c : this.mTargetClasses) {
			output += c.toString() + "\n";
		}
		return output;
	}
}
