/**
 *
 */
package de.upb.pga3.panda2.core.services;

import java.io.File;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.xmlpull.v1.XmlPullParser;

import android.content.res.AXmlResourceParser;
import android.util.TypedValue;

/**
 * Abstract class for processing XML file
 *
 * @author nptsy
 */
public abstract class BinaryParser {

	// logger
	private static final Logger LOGGER = LogManager.getLogger(ManifestXMLParser.class);

	/**
	 * Constructor
	 *
	 * @param inAPKFile
	 */
	public BinaryParser(final String inAPKFile) {
		parseData(inAPKFile);
	}

	/**
	 * parse data in .apk file
	 *
	 * @param inAPKFile
	 *            apk file path
	 * @return
	 */
	protected boolean parseData(final String inAPKFile) {
		return false;
	}

	/**
	 * get xml string from binary data
	 *
	 * @param apkPath
	 * @param inXMLFileName
	 * @return
	 */
	protected String getStrXMLFromAPK(final File apkFile, final String inXMLFileName) {
		final StringBuilder xmlSb = new StringBuilder(100);

		try {

			final ZipFile file = new ZipFile(apkFile, ZipFile.OPEN_READ);
			final ZipEntry entry = file.getEntry(inXMLFileName);

			final AXmlResourceParser parser = new AXmlResourceParser();
			parser.open(file.getInputStream(entry));

			final StringBuilder sb = new StringBuilder(10);
			final String indentStep = "	";

			int type;
			while ((type = parser.next()) != XmlPullParser.END_DOCUMENT) {
				switch (type) {
				case XmlPullParser.START_DOCUMENT: {
					constructXMLString(xmlSb, "<?xml version=\"1.0\" encoding=\"utf-8\"?>");
					break;
				}
				case XmlPullParser.START_TAG: {
					constructXMLString(false, xmlSb, "%s<%s%s", sb, getNamespacePrefix(parser.getPrefix()),
							parser.getName());
					sb.append(indentStep);

					final int namespaceCountBefore = parser.getNamespaceCount(parser.getDepth() - 1);
					final int namespaceCount = parser.getNamespaceCount(parser.getDepth());

					for (int i = namespaceCountBefore; i != namespaceCount; ++i) {
						constructXMLString(xmlSb, "%sxmlns:%s=\"%s\"", i == namespaceCountBefore ? "  " : sb,
								parser.getNamespacePrefix(i), parser.getNamespaceUri(i));
					}

					final int size = parser.getAttributeCount();
					for (int i = 0; i != size; ++i) {
						constructXMLString(false, xmlSb, "%s%s%s=\"%s\"", " ",
								getNamespacePrefix(parser.getAttributePrefix(i)), parser.getAttributeName(i),
								getAttributeValue(parser, i));
					}
					// log("%s>",sb);
					constructXMLString(xmlSb, ">");
					break;
				}
				case XmlPullParser.END_TAG: {
					sb.setLength(sb.length() - indentStep.length());
					constructXMLString(xmlSb, "%s</%s%s>", sb, getNamespacePrefix(parser.getPrefix()),
							parser.getName());
					break;
				}
				case XmlPullParser.TEXT: {
					constructXMLString(xmlSb, "%s%s", sb, parser.getText());
					break;
				}
				}
			}
			parser.close();
		} catch (final Exception e) {
			LOGGER.info("Exception: " + e.getMessage());
			e.printStackTrace();
		}
		return xmlSb.toString();
	}

	/**
	 * get namespace prefix
	 *
	 * @param prefix
	 *            prefix of namespace
	 * @return
	 */
	private String getNamespacePrefix(final String prefix) {
		if (prefix == null || prefix.length() == 0) {
			return "";
		}
		return prefix + ":";
	}

	/**
	 * get attributes
	 *
	 * @param parser
	 *            resource parser
	 * @param index
	 *            index of type and data
	 *
	 * @return a string value of attribute
	 */
	private String getAttributeValue(final AXmlResourceParser parser, final int index) {
		final int type = parser.getAttributeValueType(index);
		final int data = parser.getAttributeValueData(index);
		if (type == TypedValue.TYPE_STRING) {
			return parser.getAttributeValue(index);
		}
		if (type == TypedValue.TYPE_ATTRIBUTE) {
			return String.format("?%s%08X", getPackage(data), data);
		}
		if (type == TypedValue.TYPE_REFERENCE) {
			return String.format("@%s%08X", getPackage(data), data);
		}
		if (type == TypedValue.TYPE_FLOAT) {
			return String.valueOf(Float.intBitsToFloat(data));
		}
		if (type == TypedValue.TYPE_INT_HEX) {
			return String.format("0x%08X", data);
		}
		if (type == TypedValue.TYPE_INT_BOOLEAN) {
			return data != 0 ? "true" : "false";
		}
		if (type == TypedValue.TYPE_DIMENSION) {
			return Float.toString(complexToFloat(data)) + DIMENSION_UNITS[data & TypedValue.COMPLEX_UNIT_MASK];
		}
		if (type == TypedValue.TYPE_FRACTION) {
			return Float.toString(complexToFloat(data)) + FRACTION_UNITS[data & TypedValue.COMPLEX_UNIT_MASK];
		}
		if (type >= TypedValue.TYPE_FIRST_COLOR_INT && type <= TypedValue.TYPE_LAST_COLOR_INT) {
			return String.format("#%08X", data);
		}
		if (type >= TypedValue.TYPE_FIRST_INT && type <= TypedValue.TYPE_LAST_INT) {
			return String.valueOf(data);
		}
		return String.format("<0x%X, type 0x%02X>", data, type);
	}

	/**
	 * get android package
	 *
	 * @param id
	 * @return
	 */
	private String getPackage(final int id) {
		if (id >>> 24 == 1) {
			return "android:";
		}
		return "";
	}

	private void constructXMLString(final StringBuilder inXmlSb, final String inFormat, final Object... inArguments) {
		// append element and attributes to current xlm string
		inXmlSb.append(String.format(inFormat, inArguments));
		// constructXMLString(true, xmlSb, format, arguments);
	}

	private void constructXMLString(final boolean inNewLine, final StringBuilder inXmlSb, final String inFormat,
			final Object... inArguments) {
		// System.out.printf(format,arguments);
		// if(newLine) System.out.println();
		inXmlSb.append(String.format(inFormat, inArguments));
		if (inNewLine) {
			inXmlSb.append("\n");
		}
	}

	// ///////////////////////////////// ILLEGAL STUFF, DONT LOOK :)

	private static float complexToFloat(final int complex) {
		return (complex & 0xFFFFFF00) * RADIX_MULTS[(complex >> 4) & 3];
	}

	private static final float RADIX_MULTS[] = { 0.00390625F, 3.051758E-005F, 1.192093E-007F, 4.656613E-010F };

	private static final String DIMENSION_UNITS[] = { "px", "dip", "sp", "pt", "in", "mm", "", "" };

	private static final String FRACTION_UNITS[] = { "%", "%p", "", "", "", "", "", "" };

}
