package de.upb.pga3.panda2.core.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import de.upb.pga3.panda2.core.datastructures.EnhancedInput;
import de.upb.pga3.panda2.core.datastructures.IntentFilter;
import de.upb.pga3.panda2.utilities.Constants;
import soot.Body;
import soot.PatchingChain;
import soot.SootClass;
import soot.Unit;
import soot.ValueBox;
import soot.jimple.internal.JAssignStmt;
import soot.jimple.internal.JIdentityStmt;
import soot.toolkits.graph.ExceptionalUnitGraph;
import soot.toolkits.graph.UnitGraph;

/**
 * This class analyzes statement existing in an input application to collect all
 * intents
 *
 * @author nptsy
 */
public final class A3StatementAnalyzer implements StatementAnalyzer {
	// Instance replied in getInstance() method. (Singleton pattern)
	private static A3StatementAnalyzer INSTANCE;

	/**
	 * Returning always the same instance of an A3StatementAnalyzerVer2 object
	 *
	 * @return DataStorage object instance
	 */
	public static A3StatementAnalyzer getInstance() {
		if (INSTANCE == null) {
			INSTANCE = new A3StatementAnalyzer();
		}
		return INSTANCE;
	}

	@Override
	public Collection<IntentInformation> getAllIntents(final Body inBody, final EnhancedInput inInput,
			final Map<String, GlobalVariable> ininMapGlobalVars) {
		final Collection<IntentInformation> collIntentInfoObjs = new ArrayList<>();

		final Collection<IntentInformation> launchedIntents = analyzeIntents(inBody, inInput, ininMapGlobalVars);
		if (launchedIntents != null && !launchedIntents.isEmpty()) {
			for (final IntentInformation intInfo : launchedIntents) {
				if (intInfo.isValid() && (intInfo.getTypeIntent() == IntentInformation.TYPE_EXPLICIT
						|| intInfo.getTypeIntent() == IntentInformation.TYPE_IMPLICIT)) {
					collIntentInfoObjs.add(intInfo);
				}
			}
		}
		return collIntentInfoObjs;
	}

	@Override
	public Collection<IntentInformation> getExplicitIntents(final Body inBody, final EnhancedInput inInput,
			final Map<String, GlobalVariable> ininMapGlobalVars) {
		final Collection<IntentInformation> returnValues = new ArrayList<>();
		final Collection<IntentInformation> launchedIntents = analyzeIntents(inBody, inInput, ininMapGlobalVars);
		if (launchedIntents != null && !launchedIntents.isEmpty()) {
			for (final IntentInformation intInfo : launchedIntents) {
				if (intInfo.getTypeIntent() == IntentInformation.TYPE_EXPLICIT && intInfo.isValid()) {
					returnValues.add(intInfo);
				}
			}
		}
		return returnValues;
	}

	@Override
	public Collection<IntentInformation> getImplicitIntents(final Body inBody, final EnhancedInput inInput,
			final Map<String, GlobalVariable> ininMapGlobalVars) {
		final Collection<IntentInformation> returnValues = new ArrayList<>();
		final Collection<IntentInformation> launchedIntents = analyzeIntents(inBody, inInput, ininMapGlobalVars);
		if (launchedIntents != null && !launchedIntents.isEmpty()) {
			for (final IntentInformation intInfo : launchedIntents) {
				if (intInfo.getTypeIntent() == IntentInformation.TYPE_IMPLICIT && intInfo.isValid()) {
					returnValues.add(intInfo);
				}
			}
		}
		return returnValues;
	}

	/**
	 * get target component by the action name specified in tag intentfilter of
	 * manifest file
	 *
	 * @param actionStr
	 *            the action name
	 * @param enhancedInput
	 *            the EnhancedInput object
	 * @return a list of soot classes which are targets of the action name
	 */
	@Override
	public Collection<SootClass> getTargetsByActionString(final String actionStr, final EnhancedInput enhancedInput) {
		final Collection<SootClass> tempResult = new ArrayList<>();
		// if the EnhancedInput is valid and has intent filter
		if (enhancedInput.getIntentFilters() != null) {
			for (final IntentFilter intentFilter : enhancedInput.getIntentFilters()) {
				if (intentFilter.getActionName().equals(actionStr)) {
					for (final SootClass classOrComponent : enhancedInput.getAppClasses()) {
						if (classOrComponent.toString().contains(intentFilter.getTargetName())) {
							tempResult.add(classOrComponent);
							// break loop because class in an app is unique
							break;
						}
					}
				}
			}
		}
		return tempResult;
	}

	/**
	 * Constructor
	 */
	private A3StatementAnalyzer() {
	}

	/**
	 * analyze valid intents
	 *
	 * @param inBody
	 * @param inInput
	 * @return
	 */
	private Collection<IntentInformation> analyzeIntents(final Body inBody, final EnhancedInput inInput,
			final Map<String, GlobalVariable> ininMapGlobalVars) {

		final Collection<IntentInformation> returnValues = new ArrayList<>();
		if (inBody != null) {
			final UnitGraph graph = new ExceptionalUnitGraph(inBody);
			final PatchingChain<Unit> chainUnt = inBody.getUnits();
			if (chainUnt != null && !chainUnt.isEmpty()) {

				/*
				 * map for storing all defined variable.
				 *
				 * key: the variable
				 *
				 * value: is a list of assignments that assign value to the
				 * variable
				 */
				final Map<String, List<Unit>> mapDefinedLocalVar = new HashMap<>();

				/*
				 * map of declared variable for intent, key is variable name and
				 * value is a unit defining the variable
				 */
				final Map<String, Unit> mapDefVarIntent = new HashMap<>();
				/*
				 * map of declared intents and their corresponding objects
				 */
				final Map<Unit, List<IntentInformation>> mapIntentInfo = new HashMap<>();

				// set of already checked units
				final Set<Unit> setCheckedUnts = new HashSet<>();

				// the first statement of the body
				final Unit firstUnt = chainUnt.getFirst();

				// list of successor of the first unit
				List<Unit> lstUnits = graph.getSuccsOf(firstUnt);
				if (lstUnits == null || lstUnits.isEmpty()) {
					return returnValues;
				}

				final List<List<Unit>> lstProcessedBranches = new ArrayList<>();
				lstProcessedBranches.add(lstUnits);

				while (!lstProcessedBranches.isEmpty()) {
					lstUnits = lstProcessedBranches.get(0);

					/*
					 * traverse the body to analyze the intent
					 */
					while (!lstUnits.isEmpty()) {
						// each unit in the list
						final Unit processedUnt = lstUnits.get(0);
						final String strProcessedUnt = processedUnt.toString();

						// in case a unit is already checked => skip
						if (setCheckedUnts.contains(processedUnt)) {
							lstUnits = removeElementAt(0, lstUnits);
							continue;
						}

						/*
						 * -----------------------------------------------------
						 * process local variables or assigned variables in case
						 * a UNIT (statement) is a JIdentityStmt or a
						 * JAssignStmt
						 */
						if (processedUnt instanceof JIdentityStmt || processedUnt instanceof JAssignStmt) {
							// get defined or assigned variable here
							final List<ValueBox> lstBoxes = processedUnt.getDefBoxes();
							if (lstBoxes != null && !lstBoxes.isEmpty()) {
								final ValueBox fiVar = lstBoxes.get(0);
								final String strFiVar = fiVar.getValue().toString();
								List<Unit> lstDefUnts = mapDefinedLocalVar.get(strFiVar);
								if (lstDefUnts == null) {
									lstDefUnts = new ArrayList<>();
									lstDefUnts.add(processedUnt);
									mapDefinedLocalVar.put(strFiVar, lstDefUnts);
								} else {
									lstDefUnts.add(processedUnt);
								}
							}
						}
						// ------------------------------------------------------------

						/*
						 * add the current checked unit to the list of already
						 * checked Unit
						 */
						setCheckedUnts.add(processedUnt);

						// remove the current checked unit out of the lstUnits
						lstUnits = removeElementAt(0, lstUnits);

						// get successor of the current checked unit
						final List<Unit> nextSucs = graph.getSuccsOf(processedUnt);
						if (nextSucs == null || nextSucs.isEmpty()) {
							continue;
						}

						/*
						 * in case of there is only one child ==> add the child
						 * to the list and continue traversing the path
						 */
						if (nextSucs.size() == 1) {
							// and those successors to the lstUnits
							lstUnits.addAll(nextSucs);
						}
						/*
						 * in case there are more than 1 child ==> add the first
						 * child to current traversed path, all other remaining
						 * child will be added to another paths
						 */
						else {
							lstUnits.add(nextSucs.get(0));
							for (int i = 1; i < nextSucs.size(); i++) {
								final List<Unit> newBranch = new ArrayList<>();
								newBranch.add(nextSucs.get(i));
								lstProcessedBranches.add(newBranch);
							}
						}

						/*
						 * -----------------------------------------------------
						 * [-->]in case a unit identifies an intent
						 */
						if (isDefinedIntent(processedUnt)) {
							final List<ValueBox> defBoxes = processedUnt.getDefBoxes();
							if (defBoxes != null && !defBoxes.isEmpty()) {
								// always get the first element in defBoxes
								final ValueBox defVar = defBoxes.get(0);
								/*
								 * add variable of new intent to the list
								 *
								 * @NOTE if a variable already exists then it is
								 * overwritten
								 */
								mapDefVarIntent.put(defVar.getValue().toString(), processedUnt);

								/*
								 * add the defined intent to the map. In case an
								 * intent is defined again => overwrite the
								 * existing definition
								 */
								final List<IntentInformation> lstTemp = new ArrayList<>();
								mapIntentInfo.put(processedUnt, lstTemp);
							}
							continue;
						}
						// [<--]-----------------------------------------------------

						/*
						 * [-->]------------------------------------------------
						 * in case a unit is a constructor of an intent ==> we
						 * can decide the type of intent (implicit or explicit)
						 */
						if (strProcessedUnt.contains(Constants.KEY_WORD_CONSTRUCTOR_INTENT)) {
							final List<IntentInformation> lstIntInfo = processIntentConstructor(processedUnt, inInput,
									mapDefinedLocalVar, ininMapGlobalVars);
							/*
							 * add the list of IntentInformation to the
							 * mapIntentInfo
							 */
							final Set<String> setVars = mapDefVarIntent.keySet();
							for (final String var : setVars) {
								final String keyword = var + ".<" + Constants.KEY_WORD_CONSTRUCTOR_INTENT;
								if (strProcessedUnt.contains(keyword)) {
									final Unit tmpUnt = mapDefVarIntent.get(var);
									// in case of valid unit
									if (tmpUnt != null) {
										final List<IntentInformation> lstTmpIntInfo = mapIntentInfo.get(tmpUnt);
										if (lstTmpIntInfo == null) {
											mapIntentInfo.put(tmpUnt, lstIntInfo);
										} else {
											lstTmpIntInfo.addAll(lstIntInfo);
										}
									}
									// break loop when finding out the defined
									// intent
									break;
								}
							}
							continue;
						}
						// [<--]-----------------------------------------------------

						/*
						 * [-->]------------------------------------------------
						 * in case a unit is a setXXX method (setAction,
						 * setClass, or setClassName) ==> we can decide the type
						 * of intent also (implicit or explicit). Currently,
						 * just process for 4 cases:
						 *
						 * setAction
						 *
						 *
						 * setClass(android.content.Context,java.lang.Class)
						 *
						 * setClassName(android.content.Context,java.lang.String
						 * ),
						 *
						 * setClassName(java.lang.String,java.lang.String)
						 */
						if (strProcessedUnt.contains(Constants.KEY_WORD_SET_ACTION_METHOD)
								|| strProcessedUnt.contains(Constants.KEY_WORD_SET_CLASS_METHOD)
								|| strProcessedUnt.contains(Constants.KEY_WORD_SET_CLASSNAME_METHOD)
								|| strProcessedUnt.contains(Constants.KEY_WORD_SET_COMPONENT)) {

							final Set<String> setVars = mapDefVarIntent.keySet();
							for (final String var : setVars) {
								final String keyword = var + ".<" + Constants.KEY_WORD_SET_METHOD;
								if (strProcessedUnt.contains(keyword)) {
									final Unit tmpUnt = mapDefVarIntent.get(var);
									// in case of valid unit
									if (tmpUnt != null) {
										final List<IntentInformation> lstTmpIntInfo = mapIntentInfo.get(tmpUnt);
										if (lstTmpIntInfo != null && !lstTmpIntInfo.isEmpty()) {
											processIntentSetMethod(processedUnt, inInput, lstTmpIntInfo,
													mapDefinedLocalVar, ininMapGlobalVars);
										}
									}

									// break loop when finding out the intent
									break;
								}
							}
							continue;
						}
						// [<--]------------------------------------------------------

						/*
						 * [-->]------------------------------------------------
						 * in case a unit is a launching Intent method (such as:
						 * startActivity, startActivities, startSerice etc.)
						 */
						String var = getIntentVariableInLaunchingMethod(strProcessedUnt);
						if (!var.isEmpty()) {
							// remove blank space at front and end of the var
							var = var.trim();
							// get defined statement of the var
							final Unit unit = mapDefVarIntent.get(var);
							/*
							 * get list of intentInformation corresponding to
							 * the statement
							 */
							final List<IntentInformation> lstInformation = mapIntentInfo.get(unit);
							if (lstInformation != null && !lstInformation.isEmpty()) {
								for (final IntentInformation intInfo : lstInformation) {
									intInfo.addStartingUnit(processedUnt);
									intInfo.setValid(true);
									if (!returnValues.contains(intInfo)) {
										returnValues.add(intInfo);
									}
								}
							}
						}
						// [-->]-----------------------------------------------------

					} // end while loop 2

					lstProcessedBranches.remove(0);
				} // end while loop 1
			}
		}

		return returnValues;
	}

	/**
	 * get target SootClass by class name String
	 *
	 * @param targetClassName
	 *            the specified target class name
	 * @param enhancedInput
	 *            the EnhancedInput for getting further information
	 * @return a list of target classes corresponding to the name
	 */
	private static Collection<SootClass> getTargetsByClassName(final String targetClassName,
			final EnhancedInput enhancedInput) {
		final Collection<SootClass> tempResult = new ArrayList<>();
		for (final SootClass component : enhancedInput.getAppClasses()) {
			if (enhancedInput.isAndroidComponent(component) && component.toString().equals(targetClassName)) {
				tempResult.add(component);
				return tempResult;
			}
		}
		return tempResult;
	}

	/**
	 * get specified target class name in a called method (statement). NOTE:
	 * since there are various type of parameter specifying for a target class,
	 * therefore it is processed by different ways
	 *
	 * @param inUnit
	 *            the statement that specifies a target class to intent
	 *
	 * @param inMapDefLocalVars
	 *            the map that defines all local variables
	 *
	 * @return a target class name which is existing in the app
	 */
	private static String getTargetClassName(final Unit unit, final Map<String, List<Unit>> inMapDefLocalVars,
			final Map<String, GlobalVariable> ininMapGlobalVars) {
		String targetClassName = null;
		String packageName = null;
		final String strUnit = unit.toString();

		/*
		 * process the case that a target class is specified directly in the
		 * constructor of the intent
		 */
		if (!strUnit.contains(Constants.METHOD_SET_CLASSNAME1) && !strUnit.contains(Constants.METHOD_SET_CLASSNAME2)) {
			// process < and >
			int pos1 = strUnit.indexOf('<');
			int pos2 = strUnit.lastIndexOf('>');

			final String strLstPrams = strUnit.substring(pos2 + 2, strUnit.length() - 1);

			String subStrUnt = strUnit.substring(pos1 + 1, pos2);
			pos1 = subStrUnt.indexOf('(');
			pos2 = subStrUnt.indexOf(')');
			subStrUnt = subStrUnt.substring(pos1 + 1, pos2);
			final String[] lstParamTypes = subStrUnt.split(",");

			int i = 0;
			if (lstParamTypes != null && lstParamTypes.length > 0) {
				for (i = 0; i < lstParamTypes.length; i++) {
					final String param = lstParamTypes[i];
					if (param.equals(Constants.KEY_WORD_CLASS)) {
						break;
					}
				}

			}
			final String[] arrPrams = strLstPrams.split(",");
			if (i >= 0 && i < arrPrams.length) {
				targetClassName = arrPrams[i];
			}
		}
		/*
		 * process for 2 cases of setClassName with 2 parameters input. One is
		 * package name and the another one is class name
		 */
		else {
			/*
			 * always get the second parameters for class and add the package
			 * name to it to make a full path of class name
			 */
			final int pos2 = strUnit.lastIndexOf('>');
			if (pos2 != -1) {
				final String strLstPrams = strUnit.substring(pos2 + 2, strUnit.length() - 1);
				if (strLstPrams != null && !strLstPrams.isEmpty()) {
					final String[] arrParams = strLstPrams.split(",");
					if (arrParams.length == 2) {
						packageName = arrParams[0].trim();
						targetClassName = arrParams[1].trim();
					}
				}
			}
		}
		targetClassName = processTargetClassName(targetClassName, packageName, inMapDefLocalVars, ininMapGlobalVars);

		return targetClassName;
	}

	/**
	 * process the target class name. It can be a referenced variable or a
	 * constant string
	 *
	 * @param inTargetClassName
	 *            the string target class name
	 * @param inPackagename
	 *            the package name of the target class
	 * @param inMapDefLocalVars
	 *            the map that defines all local variables
	 * @return a target class string
	 */
	private static String processTargetClassName(final String inTargetClassName, final String inPackagename,
			final Map<String, List<Unit>> inMapDefLocalVars, final Map<String, GlobalVariable> ininMapGlobalVars) {
		String targetClassName = inTargetClassName;
		String packagename = inPackagename;
		if (targetClassName != null && !targetClassName.isEmpty()) {
			/*
			 * in case targetClassName is a String ==> we can get directly the
			 * value here
			 */
			if (targetClassName.indexOf('"') != -1 || targetClassName.indexOf('$') == -1) {
				targetClassName = targetClassName.substring(targetClassName.indexOf('"') + 1,
						targetClassName.length() - 1);
				targetClassName = targetClassName.replaceAll("/", ".");
			}
			/*
			 * otherwise we get a variable here
			 */
			else {

				// process class name
				if (targetClassName.indexOf('$') != -1) {
					if (inMapDefLocalVars != null && !inMapDefLocalVars.isEmpty()) {
						// list of defined local variables
						final List<Unit> lstDefUnt = inMapDefLocalVars.get(targetClassName);
						if (lstDefUnt != null && !lstDefUnt.isEmpty()) {
							for (int i = lstDefUnt.size() - 1; i >= 0; i--) {

								boolean isUpdatedAlready = false;

								// the defining unit
								final String strTmpUnit = lstDefUnt.get(i).toString();
								if (strTmpUnit.contains("= \"")) {
									targetClassName = strTmpUnit.substring(strTmpUnit.indexOf('"') + 1,
											strTmpUnit.length() - 1);
									targetClassName = targetClassName.replaceAll("/", ".");
									// break loop of local vars
									break;
								} else {
									/*
									 * in case of assignment from a global
									 * variable
									 */
									if (!strTmpUnit.contains("(") && !strTmpUnit.contains(")")) {
										final Collection<GlobalVariable> colGlobeVars = ininMapGlobalVars.values();
										for (final GlobalVariable globeVar : colGlobeVars) {
											final String strGlobeVar = globeVar.toString();
											/*
											 * exactly the assignment of global
											 * to local variable
											 */
											if (strTmpUnit.contains(strGlobeVar)) {
												// list of all values
												final List<String> lstValue = globeVar.getValues();
												if (lstValue != null && !lstValue.isEmpty()) {
													for (int j = lstValue.size() - 1; j >= 0; j--) {
														final String val = lstValue.get(j);
														if (val.contains("\"")) {
															final int iPos1 = val.indexOf('"');
															final int iPos2 = val.lastIndexOf('"');
															targetClassName = val.substring(iPos1 + 1, iPos2);
															targetClassName = targetClassName.replace("/", ".");
															isUpdatedAlready = true;
															/*
															 * break loop of
															 * values
															 */
															break;
														}
													}
												}
												/*
												 * break loop of global
												 * variables
												 */
												break;
											}
										}
									}
								}
								if (isUpdatedAlready) {
									/*
									 * break loop of local value
									 */
									break;
								}

							}
						}
					}
				}

				if (packagename != null && !packagename.isEmpty()) {
					// process package name
					if (packagename.indexOf('$') != -1) {
						if (inMapDefLocalVars != null && !inMapDefLocalVars.isEmpty()) {
							final List<Unit> lstDefUnt = inMapDefLocalVars.get(packagename);
							if (lstDefUnt != null && !lstDefUnt.isEmpty()) {
								for (int i = lstDefUnt.size() - 1; i >= 0; i--) {
									final String strTmpUnit = lstDefUnt.get(i).toString();
									if (strTmpUnit.contains("= \"")) {
										packagename = strTmpUnit.substring(strTmpUnit.indexOf('"') + 1,
												strTmpUnit.length() - 1);
										packagename = packagename.replaceAll("/", ".");
									}
								}
							}
						}
					}

					packagename = packagename.replaceAll("/", ".");
					packagename = packagename.replaceAll("\"", "");

					if (!targetClassName.contains(packagename)) {
						targetClassName = packagename + "." + targetClassName;
					}
				}
			}
		}
		return targetClassName;
	}

	/**
	 * get specified action name in a called method (statement). NOTE: since the
	 * parameter ActionName is always at position 0, therefore we always get the
	 * value at position 0
	 *
	 * @param inUnit
	 *            the statement that contains the action name
	 * @param inMapDefLocalVars
	 *            the map that contains defined local variables
	 * @return the action name defined in the statement
	 */
	private static String getActionName(final Unit inUnit, final Map<String, List<Unit>> inMapDefLocalVars,
			final Map<String, GlobalVariable> ininMapGlobalVars) {
		String actionStr = null;
		final String strUnit = inUnit.toString();
		if (strUnit != null && !strUnit.isEmpty()) {

			final int pos1 = strUnit.indexOf('<');
			final int pos2 = strUnit.lastIndexOf('>');

			final String strLstPrams = strUnit.substring(pos2 + 2, strUnit.length() - 1);

			if (strLstPrams != null && !strLstPrams.isEmpty()) {
				final String[] lstPrams = strLstPrams.split(",");
				if (lstPrams != null && lstPrams.length >= 1) {
					actionStr = lstPrams[0];
				}
			}

			/*
			 * in case of actionStr is valid and is a variable ==> use the map
			 * of defined local variables
			 */
			if (actionStr != null && actionStr.contains("$")) {
				// search in the map of local variable
				if (inMapDefLocalVars != null && !inMapDefLocalVars.isEmpty()) {
					final List<Unit> lstDefUnts = inMapDefLocalVars.get(actionStr);
					if (lstDefUnts != null && !lstDefUnts.isEmpty()) {
						final int iSize = lstDefUnts.size();

						boolean isUpdatedAlready = false;

						for (int i = iSize - 1; i >= 0; i--) {
							final String strUnEle = lstDefUnts.get(i).toString();
							/*
							 * check if there is assignment of global variables.
							 * IF yes, get the value of the global variable
							 */
							if (!strUnEle.contains("(") && !strUnEle.contains(")")) {
								final Collection<GlobalVariable> colGlobeVars = ininMapGlobalVars.values();
								for (final GlobalVariable globeVar : colGlobeVars) {
									final String strGlobeVar = globeVar.toString();
									/*
									 * exactly the assignment of global to local
									 * variable
									 */
									if (strUnEle.contains(strGlobeVar)) {
										// list of all values
										final List<String> lstValue = globeVar.getValues();
										if (lstValue != null && !lstValue.isEmpty()) {
											for (final String val : lstValue) {
												if (val.contains("\"")) {
													final int iPos1 = val.indexOf('"');
													final int iPos2 = val.lastIndexOf('"');
													actionStr = val.substring(iPos1 + 1, iPos2 - 1);
													actionStr = actionStr.replace("/", ".");
													isUpdatedAlready = true;
													// break loop of values
													break;
												}
											}
										}
										// break loop of global variables
										break;
									}
								}
							}

							if (isUpdatedAlready) {
								// break loop of local variable
								break;
							}

							final int index1 = strUnEle.indexOf('"');
							final int index2 = strUnEle.indexOf('<');
							if (index2 != -1 && index1 != -1) {
								continue;
							}

							if (index2 == -1 && index1 != -1) {
								/*
								 * process for the string content value of the
								 * variable
								 */
								final int index3 = strUnEle.lastIndexOf('"');
								actionStr = strUnEle.substring(index1 + 1, index3 - 1);
								actionStr = actionStr.replace("/", ".");
								break;
							}
						}
					}
				}
			}

			// process actionStr
			if (actionStr.contains("\"")) {
				actionStr = actionStr.replaceAll("\"", "");
			}

		}

		return actionStr;
	}

	/**
	 * remove an item at a specific position in a list of unit
	 *
	 * @param inPos
	 *            position of item that will be deleted
	 * @param inLstUnits
	 *            the list that is removed an item
	 * @return the list that doesn't contain the item at the position
	 */
	private List<Unit> removeElementAt(final int inPos, final List<Unit> inLstUnits) {
		final List<Unit> lstUnits = new ArrayList<>();
		if (inLstUnits != null && !inLstUnits.isEmpty() && inPos >= 0 && inPos < inLstUnits.size()) {
			for (int i = 0; i < inLstUnits.size(); i++) {
				if (i == inPos) {
					continue;
				}
				lstUnits.add(inLstUnits.get(i));
			}
		}
		return lstUnits;
	}

	/**
	 * get input variable of intent in a launching method
	 *
	 * @param inStrUnit
	 *            the statement which is checked if it is a launching one
	 * @return an intent variable that is used the launching statement
	 */
	private String getIntentVariableInLaunchingMethod(final String inStrUnit) {
		String var = "";
		if (!inStrUnit.contains(Constants.KEY_WORD_GOTO) && !inStrUnit.contains(Constants.KEY_WORD_IF)) {
			if (inStrUnit.contains(Constants.METHOD_START_ACTIVITY)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_BUNDLE)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITIES)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITIES_BUNDLE)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_FOR_RESULT)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_FOR_RESULT_BUNDLE)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_FROM_CHILD)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_FROM_CHILD_BUNDLE)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_FROM_FRAGMENT)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_FROM_FRAGMENT_BUNDLE)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_IF_NEEDED)
					|| inStrUnit.contains(Constants.METHOD_START_ACTIVITY_IF_NEEDED_BUNDLE)
					|| inStrUnit.contains(Constants.METHOD_START_NEXT_MATCHING_ACTIVITY)
					|| inStrUnit.contains(Constants.METHOD_START_NEXT_MATCHING_ACTIVITY_BUNDLE)
					|| inStrUnit.contains(Constants.METHOD_START_SERVICE)
					|| inStrUnit.contains(Constants.METHOD_BIND_SERVICE)
					|| inStrUnit.contains(Constants.METHOD_SEND_BROADCAST_STR)
					|| inStrUnit.contains(Constants.METHOD_SEND_BROADCAST)
					|| inStrUnit.contains(Constants.METHOD_SEND_ORDERED_BROADCAST)
					|| inStrUnit.contains(Constants.METHOD_SEND_ORDERED_BROADCAST_BUNDLE)
					|| inStrUnit.contains(Constants.METHOD_SEND_STICKY_BROADCAST)
					|| inStrUnit.contains(Constants.METHOD_SEND_STICKY_ORDERED_BROADCAST)) {

				int pos1 = inStrUnit.indexOf('<');
				int pos2 = inStrUnit.indexOf('>');

				final String strLstParams = inStrUnit.substring(pos2 + 2, inStrUnit.length() - 1);

				String strParamTypes = inStrUnit.substring(pos1 + 1, pos2);
				pos1 = strParamTypes.indexOf('(');
				pos2 = strParamTypes.indexOf(')');
				strParamTypes = strParamTypes.substring(pos1 + 1, pos2);
				final String[] lstParamTypes = strParamTypes.split(",");

				/*
				 * find the position of parameter in type android.content.intent
				 */
				int i = 0;
				if (lstParamTypes != null && lstParamTypes.length > 0) {
					for (i = 0; i < lstParamTypes.length; i++) {
						final String param = lstParamTypes[i];
						if (param.equals(Constants.KEY_WORD_INTENT)) {
							break;
						}
					}

				}
				final String[] lstPrams = strLstParams.split(",");
				if (i >= 0 && i < lstPrams.length) {
					var = lstPrams[i];
				}
				return var;
			}
		}
		return var;
	}

	/**
	 * process when an intent call a set method
	 *
	 * @param inUnit
	 *            the statement that contains the call of intent
	 * @param inInput
	 *            the EnhancedInput for getting further information
	 * @param inLstIntInfo
	 *            the already defined IntentInformation got from previous steps
	 * @param inMapLocalVars
	 *            the map of already defined local variables
	 */
	private void processIntentSetMethod(final Unit inUnit, final EnhancedInput inInput,
			final List<IntentInformation> inLstIntInfo, final Map<String, List<Unit>> inMapLocalVars,
			final Map<String, GlobalVariable> ininMapGlobalVars) {
		final String strUnt = inUnit.toString();
		/*
		 * just make sure that the statement is not an if statement
		 */
		if (!strUnt.contains(Constants.KEY_WORD_GOTO) && !strUnt.contains(Constants.KEY_WORD_IF)) {

			/*
			 * set action ==> implicit intent. @Note: an implicit intent may
			 * have a target class or not
			 */
			if (strUnt.contains(Constants.METHOD_SET_ACTION)) {
				final String actionStr = getActionName(inUnit, inMapLocalVars, ininMapGlobalVars);
				if (actionStr != null) {

					for (final IntentInformation intInfo : inLstIntInfo) {
						// in case of unknown intent ==> set to implicit
						if (intInfo.getTypeIntent() == IntentInformation.TYPE_UNKNOWN) {
							intInfo.setTypeIntent(IntentInformation.TYPE_IMPLICIT);
							intInfo.setClassNameOrActionString(actionStr);
							final Collection<SootClass> collScs = getTargetsByActionString(actionStr, inInput);
							if (collScs != null && !collScs.isEmpty()) {
								intInfo.addTargetClasses(collScs);
							}
						}
					}
				}
			}
			/*
			 * setClass or setClassName ==> explicit intent. @Note explicit
			 * intents always have target classes within the same application
			 */
			else if (strUnt.contains(Constants.METHOD_SET_CLASS) || strUnt.contains(Constants.METHOD_SET_CLASSNAME1)
					|| strUnt.contains(Constants.METHOD_SET_CLASSNAME2)) {
				final String className = getTargetClassName(inUnit, inMapLocalVars, ininMapGlobalVars);
				if (className != null && !className.isEmpty()) {
					final Collection<SootClass> collScs = getTargetsByClassName(className, inInput);
					if (collScs != null && !collScs.isEmpty()) {
						for (final IntentInformation intInfo : inLstIntInfo) {
							intInfo.setClassNameOrActionString(className);
							intInfo.setTypeIntent(IntentInformation.TYPE_EXPLICIT);
							intInfo.addTargetClasses(collScs);
						}
					}
				}
			}
		}
	}

	/**
	 * process a statement which is a constructor of intent
	 *
	 * @param inUnit
	 *            the statement which calls the constructor of intent. MUST BE
	 *            ALWAYS NOT NULL
	 * @param inInput
	 *            the EnhancedInput which is used to get further information of
	 *            the statement
	 *
	 * @param inMapDefGeneralVars
	 *            the map of already defined local variables
	 *
	 * @return a list of objects IntentInformation
	 */
	private List<IntentInformation> processIntentConstructor(final Unit inUnit, final EnhancedInput inInput,
			final Map<String, List<Unit>> inMapDefGeneralVars, final Map<String, GlobalVariable> ininMapGlobalVars) {
		final String strUnt = inUnit.toString();
		final List<IntentInformation> lstIntents = new ArrayList<>();
		if (!strUnt.contains(Constants.KEY_WORD_GOTO) && !strUnt.contains(Constants.KEY_WORD_IF)) {
			/*
			 * implicit intent
			 */
			if (strUnt.contains(Constants.METHOD_CONSTRUCTOR_INTENT2)
					|| strUnt.contains(Constants.METHOD_CONSTRUCTOR_INTENT3)) {
				// 1 is implicit
				final IntentInformation intInfo = new IntentInformation(IntentInformation.TYPE_IMPLICIT);
				// get action name in the statement
				final String actionStr = getActionName(inUnit, inMapDefGeneralVars, ininMapGlobalVars);
				intInfo.setClassNameOrActionString(actionStr);
				if (actionStr != null) {
					// get target class object if any
					final Collection<SootClass> lstSC = getTargetsByActionString(actionStr, inInput);
					if (lstSC != null) {
						intInfo.addTargetClasses(lstSC);
					}
				}
				lstIntents.add(intInfo);
			}
			/*
			 * explicit intent
			 */
			else if (strUnt.contains(Constants.METHOD_CONSTRUCTOR_INTENT5)
					|| strUnt.contains(Constants.METHOD_CONSTRUCTOR_INTENT6)) {
				// 0 is explicit
				final IntentInformation intInfo = new IntentInformation(IntentInformation.TYPE_EXPLICIT);
				final String className = getTargetClassName(inUnit, inMapDefGeneralVars, ininMapGlobalVars);
				intInfo.setClassNameOrActionString(className);
				if (className != null) {
					final Collection<SootClass> lstSC = getTargetsByClassName(className, inInput);
					if (lstSC != null) {
						intInfo.addTargetClasses(lstSC);
					}
				}
				lstIntents.add(intInfo);
			}
			/*
			 * unknown intent
			 */
			else if (strUnt.contains(Constants.METHOD_CONSTRUCTOR_INTENT4)) {
				// unknown intent
				final IntentInformation intInfo = new IntentInformation(IntentInformation.TYPE_UNKNOWN);
				lstIntents.add(intInfo);
			} else {
				// normal constructor
				final IntentInformation intInfo = new IntentInformation(IntentInformation.TYPE_UNKNOWN);
				lstIntents.add(intInfo);
			}
		}
		return lstIntents;
	}

	/**
	 * check if a unit is an identifying statement for intent
	 *
	 * @param inUnit
	 *            the checked unit
	 * @return true if the unit is an identifying statement for intent,
	 *         otherwise false
	 */
	private boolean isDefinedIntent(final Unit inUnit) {

		if (inUnit != null && (inUnit instanceof JIdentityStmt || inUnit instanceof JAssignStmt)) {
			final String inStrUnit = inUnit.toString();
			// case 1: new android.content.Intent
			if (inStrUnit.contains(Constants.METHOD_DECLARED_INTENT)) {
				return true;
			}

			// case 2: @parameterX: android.content.Intent
			if (inStrUnit.contains(Constants.KEY_WORD_PARATEMER) && inStrUnit.contains(Constants.KEY_WORD_INTENT)) {
				return true;
			}

			/*
			 * case 3: assigned from a return value of another parameter or a
			 * method
			 */
			final String keyword = ": " + Constants.KEY_WORD_INTENT;
			if (inStrUnit.contains(keyword)) {
				return true;
			}
		}

		return false;
	}

	@Override
	public void extractGlobalVars(final Body inBody, final EnhancedInput inInput,
			final Map<String, GlobalVariable> mapGlobalVar) {
		final Collection<GlobalVariable> returnValues = new ArrayList<>();

		Unit unThis = null;
		String varThis = "";

		if (inBody != null) {
			final UnitGraph graph = new ExceptionalUnitGraph(inBody);
			final PatchingChain<Unit> chainUnt = inBody.getUnits();
			if (chainUnt != null && !chainUnt.isEmpty()) {

				// set of checked units
				final Set<Unit> setCheckedUnt = new HashSet<>();

				// the first unit is always [THIS] unit of the body
				unThis = chainUnt.getFirst();
				if (unThis == null) {
					return;
				}

				if (!(unThis instanceof JIdentityStmt) && !(unThis instanceof JAssignStmt)) {
					return;
				}

				// get varThis
				List<ValueBox> lstBoxes = unThis.getDefBoxes();
				if (lstBoxes != null && !lstBoxes.isEmpty()) {
					final ValueBox fiVar = lstBoxes.get(0);
					varThis = fiVar.getValue().toString();
				}

				if (varThis.isEmpty()) {
					return;
				}

				final Unit lastUn = chainUnt.getLast();

				// list of successor of the first unit
				// List<Unit> lstUnits = graph.getSuccsOf(unThis);
				List<Unit> lstUnits = graph.getPredsOf(lastUn);
				if (lstUnits == null || lstUnits.isEmpty()) {
					return;
				}

				while (!lstUnits.isEmpty()) {
					// each unit in the list
					final Unit processedUnt = lstUnits.get(0);
					final String strProcessedUnt = processedUnt.toString();

					// in case a unit is already checked => skip
					if (setCheckedUnt.contains(processedUnt)) {
						lstUnits = removeElementAt(0, lstUnits);
						continue;
					}

					/*
					 * -----------------------------------------------------
					 * process for getting defined variable or assigned variable
					 * in case a UNIT (statement) is a JIdentityStmt or a
					 * JAssignStmt
					 */
					if (processedUnt instanceof JIdentityStmt || processedUnt instanceof JAssignStmt) {
						// get defined or assigned variable here
						lstBoxes = processedUnt.getDefBoxes();

						// get used box
						final List<ValueBox> lstVB = processedUnt.getUseBoxes();

						if (lstBoxes != null && !lstBoxes.isEmpty()) {
							final ValueBox fiVar = lstBoxes.get(0);
							final String strFiVar = fiVar.getValue().toString();
							if (strFiVar.contains(varThis + ".<")) {

								final int index1 = strFiVar.indexOf('<');
								final int index2 = strFiVar.indexOf('>');
								final String subDefVar = strFiVar.substring(index1 + 1, index2);
								final String[] subPartsDefVar = subDefVar.split(" ");
								if (subPartsDefVar != null && subPartsDefVar.length == 3) {
									final SootClass sClass = inBody.getMethod().getDeclaringClass();
									if (sClass != null) {

										final GlobalVariable inMapGlobVar = mapGlobalVar.get(subPartsDefVar[2]);
										if (inMapGlobVar == null) {

											final GlobalVariable globVar = new GlobalVariable(sClass);
											globVar.setType(subPartsDefVar[1]);
											globVar.setVarName(subPartsDefVar[2]);
											globVar.addUnit(processedUnt);
											/*
											 * put the global variable to the
											 * map
											 */
											mapGlobalVar.put(subPartsDefVar[2], globVar);
										} else {
											inMapGlobVar.addUnit(processedUnt);
										}
									}
								}
							}
						}
					}
					// ------------------------------------------------------------

					// add the current checked unit to the list of
					// checkedUnit
					setCheckedUnt.add(processedUnt);

					// remove the current checked unit out of the lstUnits
					lstUnits = removeElementAt(0, lstUnits);

					// get successor of the current checked unit
					final List<Unit> nextSucs = graph.getPredsOf(processedUnt);
					if (nextSucs == null || nextSucs.isEmpty()) {
						continue;
					}

					lstUnits.addAll(nextSucs);
				} // end while loop 2
			}
		}

		for (final GlobalVariable globVar : mapGlobalVar.values()) {
			returnValues.add(globVar);
		}
	}
}
