package de.upb.pga3.panda2.core.datastructures;

import java.util.ArrayList;
import java.util.List;

import de.upb.pga3.panda2.client.core.datastructures.ManifestInfo;
import de.upb.pga3.panda2.core.services.XMLParser;

/**
 * AnalysisResult is the final result of analysis of three level
 *
 * @author nptsy
 */
public abstract class AnalysisResult {
	// list of message for the result
	protected List<Message> mLstMessages;

	// App information
	private ManifestInfo appManifestInfo;
	private String appFingerprint;
	private String appName;

	// Flag setting result type
	boolean resultIsFile;

	public AnalysisResult() {
		this.mLstMessages = new ArrayList<>();
		this.resultIsFile = false;
	}

	/**
	 * initialize global variables for the class
	 *
	 * @param xmlParser
	 *            the xml perser
	 */
	public void initialize(final XMLParser xmlParser) {
		this.appManifestInfo = xmlParser.getManifestInformation();
		this.appFingerprint = xmlParser.getFingerprint();
		this.appName = xmlParser.getAppName();
	}

	/**
	 * get analysis result in graphical representation
	 *
	 * @param inDetailLvl
	 *            detail level of result
	 * @param inFilters
	 *            filter on result
	 * @return string analysis result for graphical representation
	 */
	public abstract String getGraphicalResult(DetailLevel inDetailLvl, List<String> inFilters, boolean inShowStats);

	/**
	 * get analysis result in textual representation
	 *
	 * @param inDetailLvl
	 *            detail level of result
	 * @param inFilters
	 *            filter on result
	 * @return string analysis result for textual representation
	 */
	public abstract String getTextualResult(DetailLevel inDetailLvl, List<String> inFilters, boolean inShowStats);

	/**
	 * get list of Detail level of the analysis result
	 *
	 * @return list of detail level
	 */
	public abstract List<DetailLevel> getDetailLevels();

	/**
	 * get list of filters on result
	 *
	 * @return list of filters
	 */
	public abstract List<String> getFilters();

	/**
	 * get list of messages
	 *
	 * @return list of messages
	 */
	public List<Message> getMessages() {
		return this.mLstMessages;
	}

	/**
	 * add message for current analysis result
	 *
	 * @param inMsg
	 *            input message
	 */
	public void addMessage(final Message inMsg) {
		this.mLstMessages.add(inMsg);
	}

	/**
	 * get manifest information
	 *
	 * @return
	 */
	public ManifestInfo getAppManifestInfo() {
		return this.appManifestInfo;
	}

	/**
	 * set object ManifestInfo
	 *
	 * @param appManifestInfo
	 *            the object of ManifestInfo
	 */
	public void setAppManifestInfo(final ManifestInfo appManifestInfo) {
		this.appManifestInfo = appManifestInfo;
	}

	/**
	 * get the app finger print
	 *
	 * @return a string finger print of the app
	 */
	public String getAppFingerprint() {
		return this.appFingerprint;
	}

	/**
	 * set app finger print
	 *
	 * @param appFingerprint
	 */
	public void setAppFingerprint(final String appFingerprint) {
		this.appFingerprint = appFingerprint;
	}

	/**
	 * get name of the app
	 *
	 * @return string name
	 */
	public String getAppName() {
		return this.appName;
	}

	/**
	 * set the name of app to model
	 *
	 * @param appName
	 *            the string app name
	 */
	public void setAppName(final String appName) {
		this.appName = appName;
	}

	/**
	 * check if the result of object is from saved file
	 *
	 * @return true if loaded from a saved file. Otherwise false
	 */
	public boolean getResultIsFile() {
		return this.resultIsFile;
	}

	/**
	 * set true if the result is loaded from saved file, otherwise false
	 *
	 * @param resultIsFile
	 *            the value of true or false
	 */
	public void setResultIsFile(final boolean resultIsFile) {
		this.resultIsFile = resultIsFile;
	}
}
