package de.upb.pga3.panda2.core;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.file.Path;
import java.util.Collections;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.core.services.ConfigManager;
import soot.Main;
import soot.PackManager;
import soot.Scene;
import soot.SootMethod;
import soot.Transform;
import soot.options.Options;

/**
 * Configure Soot framework
 *
 * @author npts
 *
 */
public final class SootAdapter {

	private static final Logger LOGGER = LogManager.getLogger(SootAdapter.class);

	private PrintStream systemOut;

	/**
	 * the specific phase what will be used in Soot framework
	 */
	public enum SootPhase {
		WJTP, JTP
	}

	private static SootAdapter instance;

	/**
	 * get current object SootAdapter
	 *
	 * @return
	 */
	public static SootAdapter getInstance() {
		if (instance == null) {
			instance = new SootAdapter();
		}
		return instance;
	}

	/**
	 * Constructor
	 */
	private SootAdapter() {

	}

	/**
	 * set entry point to soot. This is used for creating call graph
	 *
	 * @param entryPoint
	 *            the dummy main method
	 *
	 */
	public void setEntryPoint(final SootMethod entryPoint) {

		Scene.v().setEntryPoints(Collections.singletonList(entryPoint));
		if (Scene.v().containsClass(entryPoint.getDeclaringClass().getName())) {
			Scene.v().removeClass(entryPoint.getDeclaringClass());
		}
		Scene.v().addClass(entryPoint.getDeclaringClass());
	}

	/**
	 * reset soot
	 */
	public void reinitSoot() {
		disableOutput();
		try {

			soot.G.reset();

		} finally {
			enableOutput();
		}
	}

	/**
	 * init parameters for running soot
	 *
	 * @param apkFile
	 *            path of apk file
	 */
	public void init(final Path apkFile) {
		disableOutput();
		try {
			Options.v().set_no_bodies_for_excluded(true);
			Options.v().set_allow_phantom_refs(true);
			Options.v().set_output_format(Options.output_format_none);
			Options.v().set_whole_program(true);

			// set excluded packages
			final List<String> excludedPacks = ConfigManager.getInstance().getExcludedPackages();

			Options.v().set_exclude(excludedPacks);

			// set .apk file path
			Options.v().set_process_dir(Collections.singletonList(apkFile.toString()));

			// set android jar path
			// Options.v().set_soot_classpath(ConfigManager.getInstance().getAndroidJarPath());
			// Options.v().set_android_jars(ConfigManager.getInstance().getAndroidJarPath());
			Options.v().set_force_android_jar(ConfigManager.getInstance().getAndroidJarPath());

			Options.v().set_src_prec(Options.src_prec_apk);
			Main.v().autoSetOptions();

			// Options.v().setPhaseOption("cg.spark", "on");
			// Options.v().setPhaseOption("cg.spark", "rta:true");
			// Options.v().setPhaseOption("cg.spark", "vta:true");

			// Load whatever we need
			Scene.v().loadNecessaryClasses();

		} finally {
			enableOutput();
		}
	}

	/**
	 * run specific phase of soot or run all
	 *
	 * @param phase
	 *            specific phase user wants to run
	 */
	public void run(final SootPhase phase) {
		if (phase == SootPhase.WJTP) {
			// just run to wjtp phase
			PackManager.v().getPack("wjpp").apply();
			PackManager.v().getPack("cg").apply();
			PackManager.v().getPack("wjtp").apply();
		} else {
			// run to jtp phase
			if (phase == SootPhase.JTP) {

				PackManager.v().runPacks();
			}
			// run all phases
			else {
				PackManager.v().runPacks();
			}
		}
		LOGGER.info("Run " + phase.toString() + " phase");
	}

	/**
	 * add defined transformer to soot
	 *
	 * @param transformer
	 *            user defined transformer
	 * @param phase
	 *            phase that users want to add the defined transformer to
	 */
	public void addTransformer(final Transform transformer, final SootPhase phase) {
		if (phase == SootPhase.WJTP) {
			PackManager.v().getPack("wjtp").add(transformer);
		} else {
			if (phase == SootPhase.JTP) {
				PackManager.v().getPack("jtp").add(transformer);
			} else {
				LOGGER.info("Not support this " + phase.toString() + " phase");
			}
		}
	}

	private void disableOutput() {
		this.systemOut = System.out;
		System.setOut(new PrintStream(new OutputStream() {
			@Override
			public void write(final int arg0) throws IOException {
				// do nothing
			}
		}));
	}

	private void enableOutput() {
		System.setOut(this.systemOut);
	}
}
