package de.upb.pga3.panda2.core;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import de.upb.pga3.panda2.core.datastructures.AnalysisResult;

/**
 * Contains the logic to perform an analysis on a single apk file or to
 * aggregate multiple sub-results depending on the analysis defined in the
 * {@code AnalysisProcedure}.
 *
 * @see AnalysisProcedure
 * @author Fabian
 *
 */
public class Analysis {

	private final boolean aggregates;

	private AnalysisResult prevRes;
	private final Path apk;
	private final AnalysisProcedure ap;

	private List<AnalysisResult> subResults;

	/**
	 * Creates an {@code Analysis} instance. The behavior depends on the
	 * parameters.
	 *
	 * @param apkFile
	 *            Path to the apk file to be analyzed. If {@code null} then the
	 *            analysis will be of aggregation type.
	 * @param prevRes
	 *            A previous {@code AnalysisResult} for {@code COMAPRISON} mode.
	 *            If {@code null} then the analysis will be performed in
	 *            {@code SUMMARY} mode.
	 * @param aProc
	 *            The {@code AnalysisProcedure} containing the analysis logic
	 *            for initial and aggregation analysis. Must not be {@code null}
	 *            .
	 */
	public Analysis(final Path apkFile, final AnalysisResult prevRes, final AnalysisProcedure aProc) {

		if (aProc == null) {
			throw new IllegalArgumentException("The AnalysisProcedure must not be null!");
		}

		this.prevRes = prevRes;
		this.apk = apkFile;
		this.ap = aProc;

		this.aggregates = (apkFile == null) ? true : false;
	}

	/**
	 * Sets the sub-results for this {@code Analysis} when of type aggregation.
	 *
	 * @param results
	 *            A list of {@link AnalysisResult}s that were computed
	 *            beforehand.
	 * @see AnalysisRunner#analyze(List)
	 */
	public void provideSubResults(final List<AnalysisResult> results) {
		this.subResults = results;
	}

	/**
	 * States if the {@code Analysis} is of type aggregation
	 *
	 * @return {@code true} if the {@code Analysis} is of type aggregation and
	 *         {@code false} if of type initial.
	 */
	public boolean aggregates() {
		return this.aggregates;
	}

	/**
	 * Performs the analysis specified in the {@code AnalysisProcedure}
	 * depending on the type of analysis (aggregation or initial). Additionally
	 * a parameter check on the given apk file and the sub-results. If the check
	 * fails, an {@code UnsupportedOperationException} will be thrown.
	 *
	 * @return The {@code AnalysisResult} originating from that analysis.
	 */
	public AnalysisResult doAnalysis() {

		if (this.aggregates) {
			if (this.subResults != null && this.subResults.size() > 0) {
				return this.ap.doAggregationAnalysis(this.subResults, this.prevRes);
			}
			throw new UnsupportedOperationException(
					"A list of subresults has not been provided beforehand or given list is empty!");
		} else {
			if (this.apk != null && Files.exists(this.apk) && !Files.isDirectory(this.apk)) {
				return this.ap.doInitialAnalysis(this.apk, this.prevRes);
			}
			throw new UnsupportedOperationException(
					"An apk was not provided or the provided apk does not exist or is not a file!");
		}
	}

	public AnalysisResult getPreviousAnalysisResult() {
		return this.prevRes;
	}

	public void setPreviousAnalysisResult(final AnalysisResult prevResult) {
		this.prevRes = prevResult;
	}
}
