package de.upb.pga3.panda2.client.core;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import de.upb.pga3.panda2.client.core.datastructures.AnalysisKey;
import de.upb.pga3.panda2.client.core.datastructures.ExtraAnalysisInput;
import de.upb.pga3.panda2.core.AnalysisFactory;
import de.upb.pga3.panda2.core.datastructures.AnalysisResult;
import de.upb.pga3.panda2.extension.lvl1.AnalysisFactoryLvl1;
import de.upb.pga3.panda2.extension.lvl1.AnalysisResultLvl1;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisFactoryLvl2a;
import de.upb.pga3.panda2.extension.lvl2a.AnalysisResultLvl2a;
import de.upb.pga3.panda2.extension.lvl2b.AnalysisFactoryLvl2b;
import de.upb.pga3.panda2.extension.lvl2b.AnalysisResultLvl2b;
import de.upb.pga3.panda2.extension.lvl2b.ExtraAnalysisInputLvl2b;

/**
 * In this class all available analyses have to be defined in oder to execute
 * them.
 *
 * @author Felix
 */
public final class AnalysisRegistry {
	private static final AnalysisRegistry INSTANCE = new AnalysisRegistry();

	public static final String LEVEL1 = "Permission Usage (Intra App - Level 1)";
	public static final String LEVEL2A = "Information Flow (Intra App - Level 2a)";
	public static final String LEVEL2B = "Permission Usage (Inter App - Level 2b)";
	public static final String UNKNOWN = "Unknown";

	List<AnalysisKey> list = new ArrayList<>();

	/**
	 * This constructor should be adjusted in order to add or remove an analysis
	 * from the client.
	 */
	private AnalysisRegistry() {
		this.list.add(new AnalysisKey(LEVEL1, new String[] { "1", "l1", "level1", "level 1", "INTRA-APP-PERMISSION",
				"INTRA-PERM", "INTRA-APP-PERM" }));
		this.list.add(new AnalysisKey(LEVEL2A, new String[] { "2a", "l2a", "level2a", "level 2a", "INTRA-APP-INFO-FLOW",
				"INTRA-INFO-FLOW", "INTRA-IF" }));
		this.list.add(new AnalysisKey(LEVEL2B, new String[] { "2b", "l2b", "level2b", "level 2b",
				"INTER-APP-PERMISSION", "INTER-APP-PERM", "INTER-PERM" }));
	}

	public static AnalysisRegistry getInstance() {
		return INSTANCE;
	}

	/**
	 * In order to add or remove an analysis the {@link AnalysisFactory} has to
	 * be created and return in this method.
	 *
	 * @param name
	 *            Name of analysis. Should be defined by a constant in this
	 *            class.
	 * @param apkFile
	 *            File referring to the .apk file of the analyzed App.
	 * @param previousResult
	 *            A previous {@link AnalysisResult}.
	 * @param apkComparison
	 *            Compare the analyzed App to the App defined in this file.
	 * @param additionalApks
	 *            Non-native Apps are defined by this variable.
	 * @param extraInput
	 *            If the analysis required any sort of extra input it is defined
	 *            here.
	 * @return The {@link AnalysisFactory} of the specific analysis.
	 */
	public AnalysisFactory getFactory(final String name, final File apkFile, final AnalysisResult previousResult,
			final File apkComparison, final List<File> additionalApks, final ExtraAnalysisInput extraInput) {
		if (name == LEVEL1) {
			if (previousResult == null) {
				return new AnalysisFactoryLvl1(apkFile, apkComparison);
			} else if (previousResult instanceof AnalysisResultLvl1) {
				return new AnalysisFactoryLvl1(apkFile, (AnalysisResultLvl1) previousResult);
			}
		} else if (name == LEVEL2A) {
			if (previousResult == null) {
				if (apkComparison != null) {
					return new AnalysisFactoryLvl2a(apkFile.toPath(), apkComparison.toPath());
				} else {
					final Path path = null;
					return new AnalysisFactoryLvl2a(apkFile.toPath(), path);
				}
			} else if (previousResult instanceof AnalysisResultLvl2a) {
				return new AnalysisFactoryLvl2a(apkFile.toPath(), (AnalysisResultLvl2a) previousResult);
			}
		} else if (name == LEVEL2B) {
			if (previousResult == null) {
				return new AnalysisFactoryLvl2b(apkFile, additionalApks, apkComparison,
						((ExtraAnalysisInputLvl2b) extraInput).getAllMode());
			} else if (previousResult instanceof AnalysisResultLvl2b) {
				return new AnalysisFactoryLvl2b(apkFile, additionalApks, (AnalysisResultLvl2b) previousResult,
						((ExtraAnalysisInputLvl2b) extraInput).getAllMode());
			}
		}

		return null;
	}

	/**
	 * Getting the name of an analysis by its name or any shortcut.
	 */
	public String getName(final String shortcutOrName) {
		for (final AnalysisKey key : this.list) {
			if (key.getName().equalsIgnoreCase(shortcutOrName) || key.getShortcuts().contains(shortcutOrName)) {
				return key.getName();
			} else {
				for (final String shortcut : key.getShortcuts()) {
					if (shortcut.equalsIgnoreCase(shortcutOrName)) {
						return key.getName();
					}
				}
			}
		}
		return UNKNOWN;
	}

	/**
	 * Getting the name of the analysis associated with the
	 * {@link AnalysisResult}.
	 */
	public String getName(final AnalysisResult loadedResult) {
		if (loadedResult instanceof AnalysisResultLvl1) {
			return LEVEL1;
		} else if (loadedResult instanceof AnalysisResultLvl2a) {
			return LEVEL2A;
		} else if (loadedResult instanceof AnalysisResultLvl2b) {
			return LEVEL2B;
		}
		return UNKNOWN;
	}

	/**
	 * Get a list of all available analyses by their names.
	 */
	public List<String> getAnalysisNames() {
		final List<String> tempList = new ArrayList<>();
		for (final AnalysisKey key : this.list) {
			tempList.add(key.getName());
		}
		return tempList;
	}
}
