package de.upb.pga3.panda2.client.cmdline;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;

import javafx.application.Application;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import de.upb.pga3.panda2.client.core.Client;
import de.upb.pga3.panda2.client.core.ClientCommandLine;
import de.upb.pga3.panda2.client.core.datastructures.UIMessage;
import de.upb.pga3.panda2.client.core.datastructures.UIMessageType;
import de.upb.pga3.panda2.client.gui.GUIView;
import de.upb.pga3.panda2.core.datastructures.DetailLevel;
import de.upb.pga3.panda2.core.datastructures.Message;
import de.upb.pga3.panda2.core.datastructures.MessageType;
import de.upb.pga3.panda2.extension.lvl1.AnalysisResultLvl1;
import de.upb.pga3.panda2.extension.lvl2b.AnalysisResultLvl2b;
import de.upb.pga3.panda2.utilities.Constants;

/**
 * This class is used for handling the functionality of the application, when
 * the user perform the request to our application using command line parameters
 *
 * @author Sriram
 *
 */

public class CommandLine {

	static final Logger LOGGER = LogManager.getLogger(CommandLine.class);
	final Scanner scan = new Scanner(System.in);

	/*
	 * Variable for accessing the business logic class and its methods for the
	 * command line scenario.
	 */
	private ClientCommandLine clientCommandLine;

	/**
	 * Method for validating the given user input with the help of calling the
	 * method in the business logic class, by passing the same input. Based on
	 * the result of this method, the further processing of the analysis will be
	 * done by the main method.
	 *
	 * It also creates all required objects within the method e.g. the
	 * {@link ClientCommandLine}
	 *
	 * @param args
	 *            The String[] represents the command line parameters give by
	 *            the user.
	 *
	 * @return boolean which represents whether the user input is valid (return
	 *         value = true) or not (return value = false).
	 */
	public boolean validateInitialInput(final String... args) {
		this.clientCommandLine = new ClientCommandLine();
		return this.clientCommandLine.validateCommandLineInputArguments(args);
	}

	/**
	 * Method for Validating and Comparing the input.
	 *
	 * @return void
	 */
	public void validateAndCompareInput() {
		try {
			final UIMessage uiMessage = this.clientCommandLine.compareApp();
			final boolean isTestingScenario = this.clientCommandLine.isTestingScenario();
			if (!uiMessage.getType().equals(UIMessageType.ERROR)) {
				Client.print(uiMessage.getType().name());
				Client.print(uiMessage.getBody());
				if (!isTestingScenario) {
					Client.print("Would you like to continue to proceed the analysis? Please type Y/N");

					String userSelection = null;
					if (this.scan.hasNextLine()) {
						userSelection = this.scan.nextLine();
					}

					if (!this.clientCommandLine.continueAnalysis(userSelection)) {
						// Client.print("Program is Terminated");
						LOGGER.info("Program is Terminated");
						this.scan.close();
						System.exit(0);
					}
				}
			} else {
				throw new Exception(uiMessage.getBody());
			}
		} catch (final Exception ex) {
			ex.printStackTrace();
			// Client.print("ERROR OCCURED in " + "validateAndCompareInput()");
			LOGGER.error("ERROR OCCURED in " + "validateAndCompareInput()");
			if (!getClientCommandLine().getUserInput().isTestingScenario()) {
				this.scan.close();
				System.exit(1);
			}
		}
	}

	/**
	 * Method for showing the result based on the user input by calling
	 * respective methods.
	 *
	 * @return void
	 */
	public void saveOrShowResult() {
		try {
			if (this.clientCommandLine.isHelpOptionSelected()) {
				this.scan.close();
				System.exit(1);
			} else if (this.clientCommandLine.isLoadOptionSelected()) {
				this.clientCommandLine.viewPreviousAnalysisResult();
				if (this.clientCommandLine.getAnalysisResult() == null) {
					throw new Exception("Unable to load the Analysis Result.");
				} else {
					showResult();
				}
			} else {
				if (this.clientCommandLine.isComparisonMode()) {
					validateAndCompareInput();
				}
				final UIMessage uiMessage = this.clientCommandLine.performAnalysis();
				if (!uiMessage.getType().equals(UIMessageType.ERROR)) {
					if (this.clientCommandLine.isSaveResult()) {
						this.clientCommandLine.saveResult();
					} else {
						showResult();
					}
				} else {
					throw new Exception(uiMessage.getBody());
				}
			}
		} catch (final Exception ex) {
			ex.printStackTrace();
			// Client.print("ERROR OCCURED in saveOrShowResult()");
			LOGGER.error("ERROR OCCURED in saveOrShowResult()");
			if (!getClientCommandLine().getUserInput().isTestingScenario()) {
				this.scan.close();
				System.exit(1);
			}
		}
	}

	/**
	 * Method for deciding which view show be triggered for showing the result.
	 *
	 * @return void
	 */
	public void showResult() {
		if (this.clientCommandLine.isGraphicalViewResult()) {
			showGUIGraphicalResult();
		} else if (this.clientCommandLine.isMessageViewResult()) {
			displayMessagesTextualResult(this.clientCommandLine.getMessages());
		} else {
			parseHtmlTextualResult();
			if (!this.clientCommandLine.isProgramTerminate()) {
				if (!getClientCommandLine().getUserInput().isTestingScenario()) {
					filterResult();
				}
			}
		}
	}

	/**
	 * Method for parsing the HTML result and displaying the result in the
	 * CommandLine by calling the necessary method using the instance of
	 * HTMLParser class.
	 *
	 * It also creates all required objects within the method e.g. the
	 * {@link HtmlParser}
	 *
	 * @return void
	 *
	 */
	private void parseHtmlTextualResult() {
		List<String> resultFilter = null;
		DetailLevel defaultDetailLevel;
		try {
			final List<String> initialResultFilter = this.clientCommandLine.getUserInput().getUserSelectedFilters();
			final String initialDetailLevel = this.clientCommandLine.getUserInput().getUserSelectedDetailLevel();
			if (initialResultFilter != null && !initialResultFilter.isEmpty()) {
				resultFilter = initialResultFilter;
			} else {
				final String[] prefixes = { Constants.PREFIX_PERMISSION, Constants.PREFIX_SOURCE, Constants.PREFIX_SINK };
				resultFilter = new ArrayList<>();
				if (this.clientCommandLine.getResultFilters() != null
						&& !this.clientCommandLine.getResultFilters().isEmpty()) {
					for (final String filter : this.clientCommandLine.getResultFilters()) {
						if (!filter.contains(prefixes[0]) && !filter.contains(prefixes[1])
								&& !filter.contains(prefixes[2])) {
							resultFilter.add(filter);
						}
					}
				}
				if (!resultFilter.isEmpty()) {
					if (resultFilter.contains(AnalysisResultLvl1.TREEOPTION)) {
						resultFilter.remove(AnalysisResultLvl1.TREEOPTION);
					}
					if (resultFilter.contains(AnalysisResultLvl2b.TREEOPTION)) {
						resultFilter.remove(AnalysisResultLvl2b.TREEOPTION);
					}
				} else {
					resultFilter = this.clientCommandLine.getResultFilters();
				}
			}
			if (initialDetailLevel != null) {
				defaultDetailLevel = this.clientCommandLine.getDetailLevelFromString(initialDetailLevel);
			} else {
				defaultDetailLevel = this.clientCommandLine.getDefaultDetailLevel();
			}
		} catch (final Exception e) {
			// Client.print("Selected Detail level is wrong. Default detail
			// level was selected by the application for filtering.");
			LOGGER.error("Selected Detail level is wrong. Default detail level was selected by the application for filtering.");
			defaultDetailLevel = this.clientCommandLine.getDefaultDetailLevel();
		}
		final String textualResult = this.clientCommandLine.filterTextViewResult(defaultDetailLevel, resultFilter);
		HtmlParser.parse(textualResult, this.clientCommandLine.getUserInput().isTestingScenario());
	}

	/**
	 * Method for showing the Graphical Result by triggering the GUI.
	 *
	 * @return void
	 */
	private void showGUIGraphicalResult() {
		if (this.clientCommandLine.isLoadOptionSelected()) {
			GUIView.setInitialResult(getClientCommandLine().getAnalysisResult(), new File(getClientCommandLine()
					.getUserInput().getFilePath()));
		} else {
			GUIView.setInitialResult(getClientCommandLine().getAnalysisResult(), getClientCommandLine().getUserInput()
					.getInitialInputAPKFiles().get(0));
		}
		Application.launch(GUIView.class);
	}

	/**
	 * Method for displaying the textual result of the Messages.
	 *
	 * @param messages
	 *            The List<Message> represents the analysis result messages list
	 */
	private void displayMessagesTextualResult(final List<Message> messages) {

		final boolean isTestingScenario = this.clientCommandLine.isTestingScenario();

		final List<Message> errorMessagesList = new ArrayList<>();
		final List<Message> warningMessagesList = new ArrayList<>();
		final List<Message> infoMessagesList = new ArrayList<>();
		final List<Message> suggestionsMessagesList = new ArrayList<>();

		for (final Message message : messages) {
			if (message.getType() == MessageType.SUGGESTION) {
				suggestionsMessagesList.add(message);
			} else if (message.getType() == MessageType.ERROR) {
				errorMessagesList.add(message);
			} else if (message.getType() == MessageType.WARNING) {
				warningMessagesList.add(message);
			} else {
				infoMessagesList.add(message);
			}
		}
		printTextualMessageResult(errorMessagesList, isTestingScenario);
		printTextualMessageResult(warningMessagesList, isTestingScenario);
		printTextualMessageResult(infoMessagesList, isTestingScenario);
		printTextualMessageResult(suggestionsMessagesList, isTestingScenario);
	}

	/**
	 * Method for printing the textual result of the Messages.
	 *
	 * @param messages
	 *            The List<Message> represents the analysis result messages list
	 *            isTestingScenario Boolean value represents whether the test
	 *            case is for testing or not.
	 */
	private void printTextualMessageResult(final List<Message> messages, final boolean isTestingScenario) {
		int messageCount = 0;
		for (final Message message : messages) {
			messageCount++;
			if (!isTestingScenario) {
				Client.print("\n Message:" + messageCount);
				Client.print("\n ---Message Type--- " + message.getType().toString());
				Client.print("\n ---Message Body--- " + message.getBody());
				Client.print("\n ---Message Title--- " + message.getTitle());
			}
		}
	}

	/**
	 * Get the instance of the ClientCommandLine class.
	 *
	 * @return clientCommandLine, which returns the instance of the
	 *         ClientCommandLine class.
	 */
	public ClientCommandLine getClientCommandLine() {
		return this.clientCommandLine;
	}

	/**
	 * Method for showing the filtering options in the command line and showing
	 * the filtered result in the command line.
	 *
	 * @return void
	 *
	 */
	public void filterResult() {
		try {
			final boolean isTestingScenario = this.clientCommandLine.getUserInput().isTestingScenario();
			if (!isTestingScenario) {
				Client.print("Would you like to filter the result? Please type Y/N");

				String userSelection = null;
				if (this.scan.hasNextLine()) {
					userSelection = this.scan.nextLine();
				}
				if (this.clientCommandLine.continueAnalysis(userSelection)) {
					showDetailLevelAndFilterResult();
					final DetailLevel selectedDetailLevel = selectedDetailLevel(this.scan);
					final List<String> userSelectedResultList = getUserSelectedResultList(this.scan);
					final String textualResult = this.clientCommandLine.filterTextViewResult(selectedDetailLevel,
							userSelectedResultList);
					HtmlParser.parse(textualResult, isTestingScenario);
					filterResult();
				} else if (userSelection.equalsIgnoreCase("NO") || userSelection.equalsIgnoreCase("N")) {
					this.scan.close();
				} else {
					Client.print("Invalid Input. So please select the valid input again");
					filterResult();
				}
			} else {
				final String textualResult = this.clientCommandLine.filterTextViewResult(
						this.clientCommandLine.getDefaultDetailLevel(), this.clientCommandLine.getResultFilters());
				HtmlParser.parse(textualResult, isTestingScenario);
			}
		} catch (final Exception ex) {
			// Client.print("Program was terminated. Internal error while
			// filtering the result.");
			LOGGER.error("Program was terminated. Internal error while filtering the result.");
		}
	}

	/**
	 * Method for showing the available detail levels and result filters based
	 * on the analysis result.
	 *
	 * @return void
	 */
	private void showDetailLevelAndFilterResult() {
		String availableDetailLevel = "\n ---AVAILABLE DETAIL LEVELS--- \n";
		for (final String detailLevel : this.clientCommandLine.getDetailLevels()) {
			availableDetailLevel += detailLevel + "\n";
		}
		Client.print(availableDetailLevel);
		String availableResultFilter = "\n ---AVAILABLE RESULT FILTERS--- \n";
		for (final String resultFilter : this.clientCommandLine.getResultFilters()) {
			availableResultFilter += resultFilter + "\n";
		}
		Client.print(availableResultFilter);
	}

	/**
	 * Method for fetching the selected detail level by the user based on the
	 * input
	 *
	 * @param scan
	 *            Scanner object which passes the values entered by the user in
	 *            the console
	 * @return selectedDetailLevel, DetailLevel object which was selected by the
	 *         user
	 *
	 */
	private DetailLevel selectedDetailLevel(final Scanner scan) {
		DetailLevel selectedDetailLevel = null;
		try {
			Client.print("Select your desired level:");
			String selectedString = null;
			if (scan.hasNextLine()) {
				selectedString = scan.nextLine();
			}
			selectedDetailLevel = this.clientCommandLine.getDetailLevelFromString(selectedString);
		} catch (final Exception e) {
			// Client.print("Selected Detail level is wrong. Please give the
			// correct detail level.");
			LOGGER.error("Selected Detail level is wrong. Please give the correct detail level.");
			selectedDetailLevel = selectedDetailLevel(scan);
		}
		return selectedDetailLevel;
	}

	/**
	 *
	 * Method for fetching the user selected result filters.
	 *
	 * @param scan
	 *            Scanner object which passes the values entered by the user in
	 *            the console
	 * @return userSelectedResultList, as list of String, which represents the
	 *         result filters
	 *
	 */
	private List<String> getUserSelectedResultList(final Scanner scan) {
		final List<String> userSelectedResultList = new ArrayList<>();
		Client.print("Select your desired filter:");
		while (true) {
			String line = null;
			if (scan.hasNextLine()) {
				line = scan.nextLine();
			}
			if (line != null && !line.equalsIgnoreCase("")) {
				userSelectedResultList.add(line);
			} else {
				break;
			}
		}
		return userSelectedResultList;
	}
}
